<?php
/**
 * Admin Loyalty & Memberships API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';

requireAdmin();

try {
    $pdo = getDB();
    
    // Get loyalty statistics
    $stmt = $pdo->query("
        SELECT 
            COUNT(*) as total_members,
            COALESCE(SUM(points), 0) as total_points,
            COUNT(CASE WHEN tier = 'gold' THEN 1 END) as gold_members,
            COUNT(CASE WHEN tier = 'premium' THEN 1 END) as premium_members
        FROM loyalty_points
    ");
    $loyalty_stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get filter parameters
    $tier = $_GET['tier'] ?? 'all';
    $page = (int)($_GET['page'] ?? 1);
    $limit = (int)($_GET['limit'] ?? 100);
    $offset = ($page - 1) * $limit;
    
    // Build query
    $where = [];
    $params = [];
    
    if ($tier !== 'all') {
        $where[] = "lp.tier = :tier";
        $params[':tier'] = $tier;
    }
    
    $whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';
    
    // Get total count
    $countStmt = $pdo->prepare("SELECT COUNT(*) as total FROM loyalty_points lp {$whereClause}");
    $countStmt->execute($params);
    $totalResult = $countStmt->fetch(PDO::FETCH_ASSOC);
    $total = (int)($totalResult['total'] ?? 0);
    
    // Get loyalty points
    $stmt = $pdo->prepare("
        SELECT 
            lp.*,
            c.email as customer_email,
            c.full_name as customer_name
        FROM loyalty_points lp
        INNER JOIN customers c ON lp.customer_id = c.id
        {$whereClause}
        ORDER BY lp.points DESC
        LIMIT :limit OFFSET :offset
    ");
    
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $loyalty_points = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get memberships
    $membershipStmt = $pdo->query("
        SELECT 
            m.*,
            c.email as customer_email,
            c.full_name as customer_name
        FROM memberships m
        INNER JOIN customers c ON m.customer_id = c.id
        ORDER BY m.created_at DESC
        LIMIT 50
    ");
    $memberships = $membershipStmt->fetchAll(PDO::FETCH_ASSOC);
    
    sendSuccess([
        'loyalty_stats' => $loyalty_stats,
        'loyalty_points' => $loyalty_points,
        'memberships' => $memberships,
        'pagination' => [
            'total' => $total,
            'page' => $page,
            'limit' => $limit,
            'pages' => ceil($total / $limit)
        ]
    ]);
} catch (Exception $e) {
    error_log('Admin Loyalty API Error: ' . $e->getMessage());
    sendError('Failed to load loyalty data', 500);
}

