<?php
/**
 * Admin Events API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';
require_once __DIR__ . '/../../includes/permissions.php';

requireAdmin();

// Check permission to view events
if (!hasPermission('events.view') && !isSuperAdmin()) {
    sendError('You do not have permission to view events.', 403);
}

try {
    $pdo = getDB();
    
    // Get filter parameters
    $status = $_GET['status'] ?? 'all';
    $search = $_GET['search'] ?? '';
    $page = (int)($_GET['page'] ?? 1);
    $limit = (int)($_GET['limit'] ?? 100);
    $offset = ($page - 1) * $limit;
    
    // Build query
    $where = [];
    $params = [];
    
    if ($status !== 'all') {
        $where[] = "e.status = :status";
        $params[':status'] = $status;
    }
    
    if ($search) {
        $where[] = "(e.name LIKE :search OR e.description LIKE :search)";
        $params[':search'] = '%' . $search . '%';
    }
    
    $whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';
    
    // Get total count
    $countStmt = $pdo->prepare("SELECT COUNT(*) as total FROM events e {$whereClause}");
    $countStmt->execute($params);
    $totalResult = $countStmt->fetch(PDO::FETCH_ASSOC);
    $total = (int)($totalResult['total'] ?? 0);
    
    // Get events
    $stmt = $pdo->prepare("
        SELECT 
            e.*,
            o.company_name as organizer_name,
            v.venue_name
        FROM events e
        LEFT JOIN organizers o ON e.organizer_id = o.id
        LEFT JOIN venues v ON e.venue_id = v.id
        {$whereClause}
        ORDER BY e.created_at DESC
        LIMIT :limit OFFSET :offset
    ");
    
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $events = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    sendSuccess([
        'events' => $events,
        'pagination' => [
            'total' => $total,
            'page' => $page,
            'limit' => $limit,
            'pages' => ceil($total / $limit)
        ]
    ]);
} catch (Exception $e) {
    error_log('Admin Events API Error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
    sendError('Failed to load events: ' . $e->getMessage(), 500);
}

