<?php
/**
 * Admin Dashboard API
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/api_helper.php';
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/roles.php';

requireAdmin();

try {
    $pdo = getDB();
    
    // Get statistics
    $stats = [];
    
    // Total events
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM events");
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $stats['total_events'] = (int)($result['total'] ?? 0);
    
    // Total users
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM customers WHERE is_active = 1");
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $stats['total_users'] = (int)($result['total'] ?? 0);
    
    // Total organizers
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM organizers WHERE is_active = 1");
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $stats['total_organizers'] = (int)($result['total'] ?? 0);
    
    // Total venues
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM venues WHERE is_active = 1");
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $stats['total_venues'] = (int)($result['total'] ?? 0);
    
    // Check if deleted_at column exists
    try {
        $checkColumn = $pdo->query("SHOW COLUMNS FROM events LIKE 'deleted_at'");
        $hasDeletedAt = $checkColumn->rowCount() > 0;
    } catch (Exception $e) {
        $hasDeletedAt = false;
    }
    
    $deletedAtFilter = $hasDeletedAt ? "AND deleted_at IS NULL" : "";
    
    // Event status counts
    $statusCounts = [];
    $statuses = ['published', 'draft', 'cancelled', 'pending', 'approved', 'live', 'completed'];
    
    foreach ($statuses as $status) {
        $stmt = $pdo->query("SELECT COUNT(*) as total FROM events WHERE status = '$status' $deletedAtFilter");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $statusCounts[$status] = (int)($result['total'] ?? 0);
    }
    
    $stats['published_events'] = $statusCounts['published'] + $statusCounts['approved'] + $statusCounts['live'];
    $stats['draft_events'] = $statusCounts['draft'];
    $stats['cancelled_events'] = $statusCounts['cancelled'];
    $stats['pending_events'] = $statusCounts['pending'];
    
    // Total revenue
    $stmt = $pdo->query("SELECT SUM(final_amount) as total FROM bookings WHERE payment_status = 'completed'");
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $stats['total_revenue'] = (float)($result['total'] ?? 0);
    
    // Event performance over time (last 14 days)
    $performanceData = [];
    for ($i = 13; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-$i days"));
        $stmt = $pdo->query("SELECT COUNT(*) as total FROM events WHERE DATE(created_at) = '$date' $deletedAtFilter");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $count = (int)($result['total'] ?? 0);
        
        $performanceData[] = [
            'date' => date('M j', strtotime($date)),
            'count' => $count
        ];
    }
    
    // Event status distribution for pie chart (Red theme colors)
    $statusDistribution = [
        ['label' => 'Published', 'value' => $stats['published_events'], 'color' => '#dc2626'],
        ['label' => 'Draft', 'value' => $stats['draft_events'], 'color' => '#f87171'],
        ['label' => 'Cancelled', 'value' => $stats['cancelled_events'], 'color' => '#991b1b'],
        ['label' => 'Pending', 'value' => $stats['pending_events'], 'color' => '#fca5a5']
    ];
    
    // Recent events - Get all events (SELECT * FROM events WHERE 1)
    $stmt = $pdo->query("
        SELECT e.*, o.company_name as organizer_name
        FROM events e
        LEFT JOIN organizers o ON e.organizer_id = o.id
        WHERE 1 $deletedAtFilter
        ORDER BY e.created_at DESC
        LIMIT 10
    ");
    $recent_events = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Log for debugging
    error_log('Dashboard API: Found ' . count($recent_events) . ' events');
    
    sendSuccess([
        'stats' => $stats,
        'status_distribution' => $statusDistribution,
        'performance_data' => $performanceData,
        'recent_events' => $recent_events
    ]);
} catch (Exception $e) {
    error_log('Admin Dashboard API Error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
    sendError('Failed to load dashboard data: ' . $e->getMessage(), 500);
}

