/**
 * Admin Wallet Page - API-based
 */

document.addEventListener('DOMContentLoaded', function() {
    console.log('Wallet page loaded');
    console.log('ADMIN_API_BASE:', typeof ADMIN_API_BASE !== 'undefined' ? ADMIN_API_BASE : 'NOT DEFINED');
    
    // Small delay to ensure admin-api.js is fully loaded
    setTimeout(function() {
        loadWalletData();
    }, 100);
});

function loadWalletData() {
    // Function body

    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE is not defined!');
        showError('API configuration error. Please refresh the page.');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + 'walletapi.php';
    console.log('Loading wallet data from:', apiUrl);
    showLoading('walletTableContainer');
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => {
        console.log('Wallet API Response status:', response.status);
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            return response.text().then(text => {
                console.error('Non-JSON response:', text.substring(0, 500));
                throw new Error('Server returned non-JSON response');
            });
        }
        if (!response.ok) {
            return response.json().then(data => {
                throw new Error(data.message || `HTTP error! status: ${response.status}`);
            });
        }
        return response.json();
    })
    .then(data => {
        console.log('Wallet API Response:', data);
        hideLoading('walletTableContainer');
        if (data.success && data.data) {
            populateWalletStats(data.data.wallet_stats || {});
            populateTransactionsTable(data.data.transactions || []);
        } else {
            console.error('API returned error:', data);
            showError(data.message || 'Failed to load wallet data');
        }
    })
    .catch(error => {
        hideLoading('walletTableContainer');
        console.error('Wallet API Error:', error);
        showError('Error loading wallet data: ' + error.message);
    });
}

// Make function globally available
window.loadWalletData = loadWalletData;

function populateWalletStats(stats) {
    updateStatCard('total_wallets', stats.total_wallets || 0);
    updateStatCard('total_balance', stats.total_balance || 0, true);
    updateStatCard('active_wallets', stats.active_wallets || 0);
}

function populateTransactionsTable(transactions) {
    const tbody = document.querySelector('#transactionsTable tbody');
    if (!tbody) return;
    
    if (transactions.length === 0) {
        tbody.innerHTML = '<tr><td colspan="7" style="text-align: center; padding: 2rem;">No transactions found</td></tr>';
        return;
    }
    
    tbody.innerHTML = transactions.map(txn => `
        <tr>
            <td>${txn.id}</td>
            <td>${escapeHtml(txn.customer_name || txn.customer_email || 'N/A')}</td>
            <td>
                <span class="status-badge status-${txn.transaction_type === 'credit' ? 'approved' : 'pending'}">
                    ${capitalizeFirst(txn.transaction_type)}
                </span>
            </td>
            <td>${formatCurrency(txn.amount)}</td>
            <td>${formatCurrency(txn.balance_after || 0)}</td>
            <td>${escapeHtml(txn.description || 'N/A')}</td>
            <td>${formatDate(txn.created_at)}</td>
        </tr>
    `).join('');
}

function updateStatCard(statId, value, isCurrency = false) {
    const elements = document.querySelectorAll(`[data-stat="${statId}"]`);
    elements.forEach(el => {
        el.textContent = isCurrency ? formatCurrency(value) : numberFormat(value);
    });
}

