/**
 * Admin Venues Page - API-based
 */

document.addEventListener('DOMContentLoaded', function() {
    console.log('Venues page loaded');
    console.log('ADMIN_API_BASE:', typeof ADMIN_API_BASE !== 'undefined' ? ADMIN_API_BASE : 'NOT DEFINED');
    
    // Small delay to ensure admin-api.js is fully loaded
    setTimeout(function() {
        loadVenues();
        
        const filterForm = document.getElementById('filterForm');
        if (filterForm) {
            filterForm.addEventListener('submit', function(e) {
                e.preventDefault();
                loadVenues();
            });
        }
    }, 100);
});

function loadVenues() {
    // Function body

    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE is not defined!');
        showError('API configuration error. Please refresh the page.');
        return;
    }
    
    const search = document.querySelector('[name="search"]')?.value || '';
    const status = document.querySelector('[name="status"]')?.value || 'all';
    
    showLoading('venuesTableContainer');
    
    const params = buildQueryString({ search, status });
    const apiUrl = ADMIN_API_BASE + 'venuesapi.php' + (params ? '?' + params : '');
    console.log('Loading venues from:', apiUrl);
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => {
        console.log('Venues API Response status:', response.status);
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            return response.text().then(text => {
                console.error('Non-JSON response:', text.substring(0, 500));
                throw new Error('Server returned non-JSON response');
            });
        }
        if (!response.ok) {
            return response.json().then(data => {
                throw new Error(data.message || `HTTP error! status: ${response.status}`);
            });
        }
        return response.json();
    })
    .then(data => {
        console.log('Venues API Response:', data);
        hideLoading('venuesTableContainer');
        if (data.success && data.data) {
            populateVenuesTable(data.data.venues || []);
        } else {
            console.error('API returned error:', data);
            showError(data.message || 'Failed to load venues');
        }
    })
    .catch(error => {
        hideLoading('venuesTableContainer');
        console.error('Venues API Error:', error);
        showError('Error loading venues: ' + error.message);
    });
}


function populateVenuesTable(venues) {
    const tbody = document.querySelector('#venuesTable tbody');
    if (!tbody) return;
    
    if (venues.length === 0) {
        tbody.innerHTML = '<tr><td colspan="7" style="text-align: center; padding: 2rem;">No venues found</td></tr>';
        return;
    }
    
    tbody.innerHTML = venues.map(venue => `
        <tr>
            <td>${venue.id}</td>
            <td>${escapeHtml(venue.venue_name)}</td>
            <td>${escapeHtml(venue.city || 'N/A')}, ${escapeHtml(venue.country || 'N/A')}</td>
            <td>${venue.capacity || 'N/A'}</td>
            <td>${venue.total_events || 0}</td>
            <td><span class="status-badge status-${venue.verification_status}">${capitalizeFirst(venue.verification_status)}</span></td>
            <td>
                <a href="venue-detail.php?id=${venue.id}" class="btn-sm btn-primary">View</a>
            </td>
        </tr>
    `).join('');
}

