/**
 * Admin Users Page - API-based
 * Note: Initialization is handled by admin-init.js
 * This file only defines the loadUsers function
 */

// Make loadUsers globally available
function loadUsers() {
    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE is not defined!');
        showError('API configuration error. Please refresh the page.');
        return;
    }
    
    const search = document.querySelector('[name="search"]')?.value || '';
    const status = document.querySelector('[name="status"]')?.value || 'all';
    
    showLoading('usersTableContainer');
    
    const params = buildQueryString({ search, status });
    const apiUrl = ADMIN_API_BASE + 'usersapi.php' + (params ? '?' + params : '');
    console.log('Loading users from:', apiUrl);
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => {
        console.log('Users API Response status:', response.status);
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        console.log('Users API Response:', data);
        console.log('Response success:', data.success);
        console.log('Response data:', data.data);
        console.log('Users array:', data.data?.users);
        console.log('Users count:', data.data?.users?.length || 0);
        
        hideLoading('usersTableContainer');
        
        if (data.success && data.data && Array.isArray(data.data.users)) {
            const users = data.data.users || [];
            console.log('Populating users table with', users.length, 'users');
            populateUsersTable(users);
        } else {
            console.error('API returned error:', data);
            const errorMsg = data.message || 'Failed to load users';
            console.error('Error message:', errorMsg);
            showError(errorMsg);
            
            // Show error in table too
            const tbody = document.querySelector('#usersTable1 tbody');
            if (tbody) {
                tbody.innerHTML = '<tr><td colspan="10" style="text-align: center; padding: 2rem; color: #dc2626;">' + errorMsg + '</td></tr>';
            }
        }
    })
    .catch(error => {
        hideLoading('usersTableContainer');
        console.error('Users API Error:', error);
        showError('Error loading users: ' + error.message);
    });
}

let allRolesList = [];
let currentEditingUserId = null;
let currentEditingUserType = null;

// Load all roles for assignment
function loadAllRoles() {
    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE not defined');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + 'rolesapi.php?action=list';
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.data) {
            allRolesList = data.data.roles || [];
            populateRoleSelect();
        } else {
            console.error('Failed to load roles:', data);
        }
    })
    .catch(error => {
        console.error('Error loading roles:', error);
    });
}

function populateRoleSelect() {
    const roleSelect = document.getElementById('roleSelect');
    if (!roleSelect) return;
    
    // Clear existing options except the first one
    roleSelect.innerHTML = '<option value="">-- Select Role --</option>';
    
    // Show all active roles (no filtering by user type to keep it simple)
    if (!Array.isArray(allRolesList) || allRolesList.length === 0) {
        return;
    }

    allRolesList.forEach(role => {
        if (!role || !role.id || !role.name || role.is_active == 0) return;

        const option = document.createElement('option');
        option.value = role.id;
        option.textContent = role.name + (role.is_system_role ? ' (System)' : '');
        roleSelect.appendChild(option);
    });
}

function populateUsersTable(users) {
    // Use specific users table
    let table = document.getElementById('usersTable1');
    if (!table) return;

    // Use existing tbody or create one if missing
    let tbody = table.querySelector('tbody');
    if (!tbody) {
        tbody = document.createElement('tbody');
        table.appendChild(tbody);
    }

    if (!users || users.length === 0) {
        console.warn('⚠️ No users to display');
        tbody.innerHTML = '<tr><td colspan="10" style="text-align: center; padding: 2rem;">No users found</td></tr>';
        return;
    }

    try {
        console.log('Generating rows for', users.length, 'users...');
        const rows = users.map((user, index) => {
            try {
                return generateUserRow(user);
            } catch (error) {
                console.error(`Error generating row for user ${index}:`, error);
                return `<tr><td colspan="10" style="color: #dc2626;">Error displaying user ${user.id || index}</td></tr>`;
            }
        }).join('');
        
        console.log('Rows generated, length:', rows.length);
        tbody.innerHTML = rows;
        console.log('✅ Table populated successfully');
    } catch (error) {
        console.error('❌ Error populating table:', error);
        console.error('Error stack:', error.stack);
        tbody.innerHTML = '<tr><td colspan="10" style="text-align: center; padding: 2rem; color: #dc2626;">Error displaying users: ' + error.message + '</td></tr>';
    }
}

function generateUserRow(user) {
    // Helper functions - use window versions if available, otherwise define inline
    const escapeHtml = (text) => {
        if (typeof window.escapeHtml === 'function') {
            return window.escapeHtml(text);
        }
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    };
    
    const capitalizeFirst = (str) => {
        if (typeof window.capitalizeFirst === 'function') {
            return window.capitalizeFirst(str);
        }
        if (!str) return '';
        return str.charAt(0).toUpperCase() + str.slice(1).toLowerCase();
    };
    
    const formatDate = (dateString) => {
        if (typeof window.formatDate === 'function') {
            return window.formatDate(dateString);
        }
        if (!dateString) return 'N/A';
        try {
            const date = new Date(dateString);
            return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
        } catch (e) {
            return dateString;
        }
    };
    
    // Determine user type badge
    let userTypeBadge = '';
    if (user.admin_role) {
        userTypeBadge = '<span class="status-badge status-approved">Admin</span>';
    } else if (user.organizer_id) {
        userTypeBadge = '<span class="status-badge" style="background: #d1fae5; color: #065f46;">Organizer</span>';
    } else if (user.venue_id) {
        userTypeBadge = '<span class="status-badge" style="background: #e9d5ff; color: #6b21a8;">Venue</span>';
    } else {
        userTypeBadge = '<span class="status-badge status-draft">Customer</span>';
    }
    
    // Show roles
    const roles = user.roles || [];
    let rolesDisplay = '';
    if (roles.length > 0) {
        rolesDisplay = roles.map(role => {
            const roleName = escapeHtml(role.name || role.slug || 'Unknown');
            return `<span class="status-badge status-approved" style="margin-right: 0.25rem; font-size: 0.75rem;">${roleName}</span>`;
        }).join('');
    } else {
        rolesDisplay = '<span style="color: #6b7280; font-size: 0.875rem;">No roles</span>';
    }
    
    const userName = escapeHtml(user.full_name || ((user.first_name || '') + ' ' + (user.last_name || '')).trim() || 'N/A');
    const userEmail = escapeHtml(user.email || 'N/A');
    const userType = (user.user_type || 'customer').replace(/'/g, "\\'").replace(/"/g, '&quot;');
    const createdDate = formatDate(user.created_at);
    const isActive = user.is_active == 1 || user.is_active === true || user.is_active === '1';
    
    return `
        <tr>
            <td>${user.id || 'N/A'}</td>
            <td>${userName}</td>
            <td>${userEmail}</td>
            <td>${userTypeBadge}</td>
            <td>${rolesDisplay}</td>
            <td>${capitalizeFirst(user.provider || 'email')}</td>
            <td>${user.total_bookings || 0}</td>
            <td>
                ${isActive ? 
                    '<span class="status-badge status-approved">Active</span>' : 
                    '<span class="status-badge status-rejected">Inactive</span>'
                }
            </td>
            <td>${createdDate}</td>
            <td>
                <div style="display: flex; gap: 0.5rem;">
                    <button onclick="openRoleAssignmentModal(${user.id}, '${userType}')" class="btn-sm btn-primary" style="font-size: 0.75rem; cursor: pointer;">Manage Roles</button>
                </div>
            </td>
        </tr>
    `;
}

// Set up form handler when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    // Load users and roles once DOM and scripts are ready
    loadUsers();
    loadAllRoles();

    const filterForm = document.getElementById('filterForm');
    if (filterForm) {
        filterForm.addEventListener('submit', function(e) {
            e.preventDefault();
            loadUsers();
        });
    }
    
    // Role assignment modal handlers
    const closeBtn = document.getElementById('closeRoleModal');
    const cancelBtn = document.getElementById('cancelRoleModal');
    const assignBtn = document.getElementById('assignRoleBtn');
    
    if (closeBtn) {
        closeBtn.addEventListener('click', closeRoleAssignmentModal);
    }
    if (cancelBtn) {
        cancelBtn.addEventListener('click', closeRoleAssignmentModal);
    }
    if (assignBtn) {
        assignBtn.addEventListener('click', handleAssignRole);
    }
});

// Make functions globally available
window.loadUsers = loadUsers;
window.openRoleAssignmentModal = openRoleAssignmentModal;
window.removeRoleFromUser = removeRoleFromUser;

function openRoleAssignmentModal(userId, userType) {
    currentEditingUserId = userId;
    currentEditingUserType = userType || 'customer';
    
    const modal = document.getElementById('roleAssignmentModal');
    const title = document.getElementById('roleModalTitle');
    const userInfo = document.getElementById('roleModalUserInfo');
    
    // Get user info
    const userRow = document.querySelector(`#usersTable1 tbody tr:has(button[onclick*="${userId}"])`);
    let userName = 'User';
    let userEmail = '';
    
    if (userRow) {
        const cells = userRow.querySelectorAll('td');
        if (cells.length > 2) {
            userName = cells[1].textContent.trim();
            userEmail = cells[2].textContent.trim();
        }
    }
    
    title.textContent = `Manage Roles - ${userName}`;
    userInfo.innerHTML = `
        <div><strong>User:</strong> ${escapeHtml(userName)}</div>
        <div><strong>Email:</strong> ${escapeHtml(userEmail)}</div>
        <div><strong>Type:</strong> ${capitalizeFirst(currentEditingUserType)}</div>
    `;
    
    // Load current roles for this user
    loadUserRoles(userId, userType);
    
    // Populate role select based on user type
    populateRoleSelect();
    
    modal.style.display = 'flex';
    
    // Close modal when clicking outside
    modal.addEventListener('click', function(e) {
        if (e.target === modal) {
            closeRoleAssignmentModal();
        }
    });
}

function closeRoleAssignmentModal() {
    document.getElementById('roleAssignmentModal').style.display = 'none';
    currentEditingUserId = null;
    currentEditingUserType = null;
}

function loadUserRoles(userId, userType) {
    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE not defined');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + `userrolesapi.php?action=user_roles&user_id=${userId}&user_type=${userType}`;
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.data) {
            displayCurrentRoles(data.data.roles || []);
        } else {
            console.error('Failed to load user roles:', data);
            displayCurrentRoles([]);
        }
    })
    .catch(error => {
        console.error('Error loading user roles:', error);
        displayCurrentRoles([]);
    });
}

function displayCurrentRoles(roles) {
    const container = document.getElementById('currentRolesList');
    if (!container) return;
    
    if (!roles || roles.length === 0) {
        container.innerHTML = '<div style="color: #6b7280; padding: 0.5rem;">No roles assigned</div>';
        return;
    }
    
    container.innerHTML = roles.map(role => {
        const roleName = escapeHtml(role.name || role.slug || 'Unknown');
        const roleSlug = escapeHtml(role.slug || '');
        const roleId = role.id || role.role_id;
        const isSystemRole = role.is_system_role == 1 || role.is_system_role === true;
        const canRemove = roleId && !isSystemRole;
        
        return `
        <div style="display: flex; justify-content: space-between; align-items: center; padding: 0.75rem; margin-bottom: 0.5rem; background: #f9fafb; border-radius: 6px;">
            <div>
                <strong>${roleName}</strong>
                <div style="font-size: 0.75rem; color: #6b7280; margin-top: 0.25rem;">
                    <code>${roleSlug}</code>
                    ${isSystemRole ? '<span class="status-badge status-approved" style="margin-left: 0.5rem; font-size: 0.7rem;">System</span>' : ''}
                    ${!roleId ? '<span class="status-badge" style="margin-left: 0.5rem; font-size: 0.7rem; background: #fef3c7; color: #92400e;">Legacy</span>' : ''}
                </div>
            </div>
            ${canRemove ? `
            <button onclick="removeRoleFromUser(${currentEditingUserId}, ${roleId}, '${roleName.replace(/'/g, "\\'")}')" 
                    class="btn-sm" 
                    style="background: #dc2626; color: white; cursor: pointer; border: none; padding: 0.5rem 1rem; border-radius: 6px;">
                Remove
            </button>
            ` : `
            <button disabled 
                    class="btn-sm" 
                    style="background: #9ca3af; color: white; cursor: not-allowed; border: none; padding: 0.5rem 1rem; border-radius: 6px;"
                    title="${!roleId ? 'Legacy roles cannot be removed' : 'System roles cannot be removed'}">
                Remove
            </button>
            `}
        </div>
    `;
    }).join('');
}

function handleAssignRole() {
    const roleSelect = document.getElementById('roleSelect');
    const roleId = roleSelect.value;
    
    if (!roleId) {
        showError('Please select a role to assign');
        return;
    }
    
    if (!currentEditingUserId) {
        showError('User ID not found');
        return;
    }
    
    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE not defined');
        showError('API configuration error');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + 'userrolesapi.php?action=assign';
    
    const data = {
        user_id: currentEditingUserId,
        role_id: parseInt(roleId),
        user_type: currentEditingUserType
    };
    
    console.log('Assigning role:', data);
    
    fetch(apiUrl, {
        method: 'POST',
        credentials: 'same-origin',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json'
        },
        body: JSON.stringify(data)
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showSuccess(data.message || 'Role assigned successfully');
            roleSelect.value = '';
            loadUserRoles(currentEditingUserId, currentEditingUserType);
            // Reload users table to show updated roles
            loadUsers();
        } else {
            showError(data.message || 'Failed to assign role');
        }
    })
    .catch(error => {
        console.error('Error assigning role:', error);
        showError('Error assigning role: ' + error.message);
    });
}

function removeRoleFromUser(userId, roleId, roleName) {
    if (!confirm(`Are you sure you want to remove the role "${roleName}" from this user?`)) {
        return;
    }
    
    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE not defined');
        showError('API configuration error');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + `userrolesapi.php?action=remove&user_id=${userId}&role_id=${roleId}`;
    
    console.log('Removing role:', { userId, roleId });
    
    fetch(apiUrl, {
        method: 'DELETE',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showSuccess(data.message || 'Role removed successfully');
            loadUserRoles(currentEditingUserId, currentEditingUserType);
            // Reload users table to show updated roles
            loadUsers();
        } else {
            showError(data.message || 'Failed to remove role');
        }
    })
    .catch(error => {
        console.error('Error removing role:', error);
        showError('Error removing role: ' + error.message);
    });
}

// Helper functions
function capitalizeFirst(str) {
    if (!str) return '';
    return str.charAt(0).toUpperCase() + str.slice(1).toLowerCase();
}

function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function showError(message) {
    // Prevent infinite recursion - use window.showError only if it's different from this function
    if (typeof window.showError === 'function' && window.showError !== showError) {
        window.showError(message);
    } else {
        // Fallback: create error message element
        const errorDiv = document.createElement('div');
        errorDiv.className = 'error-message';
        errorDiv.style.cssText = 'background: #fee2e2; color: #dc2626; padding: 1rem; border-radius: 8px; margin-bottom: 1rem;';
        errorDiv.textContent = message;
        const content = document.querySelector('.admin-content');
        if (content) {
            content.insertBefore(errorDiv, content.firstChild);
            setTimeout(() => errorDiv.remove(), 5000);
        } else {
            alert(message);
        }
    }
}

function showSuccess(message) {
    const successDiv = document.createElement('div');
    successDiv.className = 'success-message';
    successDiv.style.cssText = 'background: #d1fae5; color: #065f46; padding: 1rem; border-radius: 8px; margin-bottom: 1rem;';
    successDiv.textContent = message;
    const content = document.querySelector('.admin-content');
    if (content) {
        content.insertBefore(successDiv, content.firstChild);
        setTimeout(() => successDiv.remove(), 5000);
    }
}

