/**
 * Admin Settlements Page - API-based
 */

document.addEventListener('DOMContentLoaded', function() {
    console.log('Settlements page loaded');
    console.log('ADMIN_API_BASE:', typeof ADMIN_API_BASE !== 'undefined' ? ADMIN_API_BASE : 'NOT DEFINED');
    
    // Small delay to ensure admin-api.js is fully loaded
    setTimeout(function() {
        loadSettlements();
    }, 100);
});

function loadSettlements() {
    // Function body

    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE is not defined!');
        showError('API configuration error. Please refresh the page.');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + 'settlementsapi.php';
    console.log('Loading settlements from:', apiUrl);
    showLoading('settlementsContainer');
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => {
        console.log('Settlements API Response status:', response.status);
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            return response.text().then(text => {
                console.error('Non-JSON response:', text.substring(0, 500));
                throw new Error('Server returned non-JSON response');
            });
        }
        if (!response.ok) {
            return response.json().then(data => {
                throw new Error(data.message || `HTTP error! status: ${response.status}`);
            });
        }
        return response.json();
    })
    .then(data => {
        console.log('Settlements API Response:', data);
        hideLoading('settlementsContainer');
        const statsEl = document.getElementById('settlementStats');
        const tableEl = document.getElementById('settlementsTableSection');
        if (statsEl) statsEl.style.display = 'grid';
        if (tableEl) tableEl.style.display = 'block';
        if (data.success && data.data) {
            populateSettlementStats(data.data.settlement_stats || {});
            populateOrganizerSettlements(data.data.organizer_settlements || []);
        } else {
            console.error('API returned error:', data);
            showError(data.message || 'Failed to load settlement data');
        }
    })
    .catch(error => {
        hideLoading('settlementsContainer');
        console.error('Settlements API Error:', error);
        showError('Error loading settlement data: ' + error.message);
    });
}


function populateSettlementStats(stats) {
    updateStatCard('total_organizers', stats.total_organizers || 0);
    updateStatCard('total_revenue', stats.total_revenue || 0, true);
    updateStatCard('platform_commission', stats.platform_commission || 0, true);
    updateStatCard('organizer_payout', stats.organizer_payout || 0, true);
}

function populateOrganizerSettlements(settlements) {
    const tbody = document.querySelector('#settlementsTable tbody');
    if (!tbody) return;
    
    if (settlements.length === 0) {
        tbody.innerHTML = '<tr><td colspan="6" style="text-align: center; padding: 2rem;">No settlement data found</td></tr>';
        return;
    }
    
    tbody.innerHTML = settlements.map(settlement => `
        <tr>
            <td>${escapeHtml(settlement.company_name || 'N/A')}</td>
            <td>${formatCurrency(settlement.total_revenue || 0)}</td>
            <td>${formatCurrency(settlement.commission || 0)}</td>
            <td>${formatCurrency(settlement.payout || 0)}</td>
            <td>${settlement.total_bookings || 0}</td>
            <td>
                <a href="organizer-settlement.php?id=${settlement.organizer_id}" class="btn-sm btn-primary">View Details</a>
            </td>
        </tr>
    `).join('');
}

function updateStatCard(statId, value, isCurrency = false) {
    const elements = document.querySelectorAll(`[data-stat="${statId}"]`);
    elements.forEach(el => {
        el.textContent = isCurrency ? formatCurrency(value) : numberFormat(value);
    });
}

