/**
 * Admin Roles & Permissions Management
 */

// Helper functions (define first)
function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function showError(message) {
    if (typeof window.showError === 'function') {
        window.showError(message);
    } else {
        alert(message);
    }
}

function showSuccess(message) {
    // Create success message
    const successDiv = document.createElement('div');
    successDiv.className = 'success-message';
    successDiv.style.cssText = 'background: #d1fae5; color: #065f46; padding: 1rem; border-radius: 8px; margin-bottom: 1rem;';
    successDiv.textContent = message;
    const content = document.querySelector('.admin-content');
    if (content) {
        content.insertBefore(successDiv, content.firstChild);
        setTimeout(() => successDiv.remove(), 5000);
    }
}

// Define global functions (these will be available immediately for onclick handlers)
window.editRole = function(roleId) {
    console.log('Editing role:', roleId);
    if (!roleId) {
        showError('Invalid role ID');
        return;
    }
    // Call openRoleModal - it will be defined by the time onclick is triggered
    if (typeof openRoleModal === 'function') {
        openRoleModal(roleId);
    } else {
        // If not available yet, wait a bit
        setTimeout(() => {
            if (typeof openRoleModal === 'function') {
                openRoleModal(roleId);
            } else {
                console.error('openRoleModal not available');
                showError('Please refresh the page');
            }
        }, 200);
    }
};

window.deleteRole = function(roleId, roleName = '') {
    const confirmMessage = roleName ? 
        `Are you sure you want to delete the role "${roleName}"? This action cannot be undone.` :
        'Are you sure you want to delete this role? This action cannot be undone.';
    
    if (!confirm(confirmMessage)) {
        return;
    }
    
    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE not defined');
        showError('API configuration error');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + `rolesapi.php?action=delete&id=${roleId}`;
    
    console.log('Deleting role:', roleId);
    
    fetch(apiUrl, {
        method: 'DELETE',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => {
        console.log('Delete response status:', response.status);
        return response.json();
    })
    .then(data => {
        console.log('Delete response:', data);
        if (data.success) {
            loadRoles();
            showSuccess(data.message || 'Role deleted successfully');
        } else {
            showError(data.message || 'Failed to delete role');
        }
    })
    .catch(error => {
        console.error('Error deleting role:', error);
        showError('Error deleting role: ' + error.message);
    });
};

document.addEventListener('DOMContentLoaded', function() {
    console.log('Roles page loaded');
    loadRoles();
    loadPermissions();
    
    // Modal handlers
    document.getElementById('createRoleBtn')?.addEventListener('click', function() {
        openRoleModal();
    });
    
    document.getElementById('closeModal')?.addEventListener('click', closeRoleModal);
    document.getElementById('cancelBtn')?.addEventListener('click', closeRoleModal);
    
    document.getElementById('roleForm')?.addEventListener('submit', handleRoleSubmit);
    
    // Auto-generate slug from name
    document.getElementById('roleName')?.addEventListener('input', function(e) {
        const name = e.target.value;
        const slugInput = document.getElementById('roleSlug');
        if (slugInput && !slugInput.dataset.manual) {
            slugInput.value = name.toLowerCase().replace(/[^a-z0-9]+/g, '_').replace(/^_+|_+$/g, '');
        }
    });
    
    document.getElementById('roleSlug')?.addEventListener('input', function(e) {
        e.target.dataset.manual = 'true';
    });
});

let allPermissions = [];
let permissionsByModule = {};

// Define loadRoles function
function loadRoles() {
    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE not defined');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + 'rolesapi.php?action=list';
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.data) {
            populateRolesTable(data.data.roles || []);
        } else {
            console.error('Failed to load roles:', data);
            showError(data.message || 'Failed to load roles');
        }
    })
    .catch(error => {
        console.error('Error loading roles:', error);
        showError('Error loading roles: ' + error.message);
    });
}

function loadPermissions() {
    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE not defined');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + 'rolesapi.php?action=permissions_by_module';
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.data) {
            permissionsByModule = data.data.permissions || {};
            allPermissions = [];
            Object.values(permissionsByModule).forEach(perms => {
                allPermissions = allPermissions.concat(perms);
            });
            renderPermissions();
        } else {
            console.error('Failed to load permissions:', data);
        }
    })
    .catch(error => {
        console.error('Error loading permissions:', error);
    });
}

function populateRolesTable(roles) {
    const tbody = document.querySelector('#rolesTable tbody');
    if (!tbody) return;
    
    if (roles.length === 0) {
        tbody.innerHTML = '<tr><td colspan="7" style="text-align: center; padding: 2rem;">No roles found</td></tr>';
        return;
    }
    
    // Check if user has edit/delete permissions (assuming these are available globally or check via API)
    const canEdit = true; // Will be checked on server side
    const canDelete = true; // Will be checked on server side
    
    tbody.innerHTML = roles.map(role => {
        const permissionCount = role.permission_count || 0;
        const isSystemRole = role.is_system_role == 1 || role.is_system_role === true || role.is_system_role == '1';
        const isActive = role.is_active == 1 || role.is_active === true || role.is_active == '1';
        
        const typeBadge = isSystemRole ? 
            '<span class="status-badge status-approved">System</span>' : 
            '<span class="status-badge status-draft">Custom</span>';
        const statusBadge = isActive ? 
            '<span class="status-badge status-approved">Active</span>' : 
            '<span class="status-badge status-rejected">Inactive</span>';
        
        let actions = '<div class="role-actions">';
        
        // Edit button - show for all roles
        actions += `<button onclick="editRole(${role.id})" class="btn-sm btn-primary" style="margin-right: 0.5rem; cursor: pointer;">Edit</button>`;
        
        // Delete button - only for custom roles (not system roles)
        if (!isSystemRole) {
            actions += `<button onclick="deleteRole(${role.id}, '${escapeHtml(role.name)}')" class="btn-sm" style="background: #dc2626; color: white; cursor: pointer;">Delete</button>`;
        } else {
            actions += '<span style="color: #6b7280; font-size: 0.875rem; padding: 0.375rem 0.75rem;">Cannot delete</span>';
        }
        
        actions += '</div>';
        
        return `
            <tr>
                <td>${role.id}</td>
                <td>${escapeHtml(role.name)}</td>
                <td><code>${escapeHtml(role.slug)}</code></td>
                <td>${typeBadge}</td>
                <td>${permissionCount} permissions</td>
                <td>${statusBadge}</td>
                <td>${actions}</td>
            </tr>
        `;
    }).join('');
}

function renderPermissions() {
    const container = document.getElementById('permissionsContainer');
    if (!container) return;
    
    container.innerHTML = '';
    
    Object.keys(permissionsByModule).sort().forEach(module => {
        const moduleDiv = document.createElement('div');
        moduleDiv.className = 'permission-group';
        
        const moduleTitle = document.createElement('h4');
        moduleTitle.textContent = capitalizeFirst(module);
        moduleDiv.appendChild(moduleTitle);
        
        const permissions = permissionsByModule[module];
        permissions.forEach(permission => {
            const permDiv = document.createElement('div');
            permDiv.className = 'permission-item';
            
            const checkbox = document.createElement('input');
            checkbox.type = 'checkbox';
            checkbox.id = `perm_${permission.id}`;
            checkbox.value = permission.id;
            checkbox.name = 'permissions[]';
            
            const label = document.createElement('label');
            label.htmlFor = `perm_${permission.id}`;
            label.style.marginLeft = '0.5rem';
            label.textContent = permission.name;
            
            permDiv.appendChild(checkbox);
            permDiv.appendChild(label);
            moduleDiv.appendChild(permDiv);
        });
        
        container.appendChild(moduleDiv);
    });
}

// Define openRoleModal function
function openRoleModal(roleId = null) {
    const modal = document.getElementById('roleModal');
    const form = document.getElementById('roleForm');
    const title = document.getElementById('modalTitle');
    const slugInput = document.getElementById('roleSlug');
    
    if (roleId) {
        title.textContent = 'Edit Role';
        // Enable slug input initially (will be disabled if system role)
        if (slugInput) {
            slugInput.disabled = false;
            slugInput.title = '';
        }
        loadRoleData(roleId);
    } else {
        title.textContent = 'Create Role';
        form.reset();
        document.getElementById('roleId').value = '';
        if (slugInput) {
            slugInput.dataset.manual = '';
            slugInput.disabled = false;
            slugInput.title = '';
        }
        // Uncheck all permissions
        setTimeout(() => {
            document.querySelectorAll('#permissionsContainer input[type="checkbox"]').forEach(cb => {
                cb.checked = false;
            });
        }, 100);
    }
    
    modal.style.display = 'flex';
}

// Define closeRoleModal function
function closeRoleModal() {
    document.getElementById('roleModal').style.display = 'none';
}

// Define loadRoleData function
function loadRoleData(roleId) {
    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE not defined');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + `rolesapi.php?action=get&id=${roleId}`;
    
    console.log('Loading role data for ID:', roleId);
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => {
        console.log('Load role response status:', response.status);
        return response.json();
    })
    .then(data => {
        console.log('Load role response:', data);
        if (data.success && data.data) {
            const role = data.data.role;
            console.log('Role data:', role);
            
            document.getElementById('roleId').value = role.id;
            document.getElementById('roleName').value = role.name || '';
            document.getElementById('roleSlug').value = role.slug || '';
            document.getElementById('roleSlug').dataset.manual = 'true';
            document.getElementById('roleDescription').value = role.description || '';
            document.getElementById('roleStatus').value = role.is_active || 1;
            
            // Disable slug for system roles
            const slugInput = document.getElementById('roleSlug');
            if (role.is_system_role == 1 || role.is_system_role === true || role.is_system_role == '1') {
                slugInput.disabled = true;
                slugInput.title = 'System role slug cannot be changed';
            } else {
                slugInput.disabled = false;
                slugInput.title = '';
            }
            
            // Wait a bit for permissions container to be ready, then check permissions
            setTimeout(() => {
                const permissionIds = role.permissions || [];
                console.log('Setting permissions for role:', permissionIds);
                console.log('Available checkboxes:', document.querySelectorAll('#permissionsContainer input[type="checkbox"]').length);
                
                document.querySelectorAll('#permissionsContainer input[type="checkbox"]').forEach(cb => {
                    const permId = parseInt(cb.value);
                    const isChecked = permissionIds.includes(permId) || 
                                     permissionIds.includes(permId.toString()) ||
                                     permissionIds.some(p => parseInt(p) === permId);
                    cb.checked = isChecked;
                });
                console.log('Permissions set');
            }, 100);
        } else {
            showError(data.message || 'Failed to load role');
        }
    })
    .catch(error => {
        console.error('Error loading role:', error);
        showError('Error loading role: ' + error.message);
    });
}

function handleRoleSubmit(e) {
    e.preventDefault();
    
    const formData = new FormData(e.target);
    const roleId = formData.get('id');
    const permissions = Array.from(document.querySelectorAll('#permissionsContainer input[type="checkbox"]:checked'))
        .map(cb => parseInt(cb.value));
    
    const data = {
        name: formData.get('name'),
        slug: formData.get('slug'),
        description: formData.get('description'),
        is_active: parseInt(formData.get('is_active')),
        permissions: permissions
    };
    
    if (roleId) {
        data.id = parseInt(roleId);
        updateRole(data);
    } else {
        createRole(data);
    }
}

function createRole(data) {
    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE not defined');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + 'rolesapi.php?action=create';
    
    fetch(apiUrl, {
        method: 'POST',
        credentials: 'same-origin',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json'
        },
        body: JSON.stringify(data)
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            closeRoleModal();
            loadRoles();
            if (typeof showSuccess === 'function') {
                showSuccess(data.message || 'Role created successfully');
            } else {
                alert(data.message || 'Role created successfully');
            }
        } else {
            showError(data.message || 'Failed to create role');
        }
    })
    .catch(error => {
        console.error('Error creating role:', error);
        showError('Error creating role: ' + error.message);
    });
}

function updateRole(data) {
    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE not defined');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + 'rolesapi.php?action=update';
    
    fetch(apiUrl, {
        method: 'PUT',
        credentials: 'same-origin',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json'
        },
        body: JSON.stringify(data)
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            closeRoleModal();
            loadRoles();
            if (typeof showSuccess === 'function') {
                showSuccess(data.message || 'Role updated successfully');
            } else {
                alert(data.message || 'Role updated successfully');
            }
        } else {
            showError(data.message || 'Failed to update role');
        }
    })
    .catch(error => {
        console.error('Error updating role:', error);
        showError('Error updating role: ' + error.message);
    });
}

// Functions are already defined globally above

// Helper functions (already defined above)
function capitalizeFirst(str) {
    if (!str) return '';
    return str.charAt(0).toUpperCase() + str.slice(1).toLowerCase();
}

