/**
 * Admin Organizers Page - API-based
 * Note: Initialization is handled by admin-init.js
 * This file only defines the loadOrganizers function
 */

function loadOrganizers() {
    // Check both ADMIN_API_BASE and window.ADMIN_API_BASE
    const apiBase = typeof ADMIN_API_BASE !== 'undefined' ? ADMIN_API_BASE : (typeof window.ADMIN_API_BASE !== 'undefined' ? window.ADMIN_API_BASE : null);
    
    if (!apiBase) {
        console.error('❌ ADMIN_API_BASE is not defined!');
        console.error('BASE_URL:', typeof BASE_URL !== 'undefined' ? BASE_URL : 'UNDEFINED');
        console.error('window.ADMIN_API_BASE:', typeof window.ADMIN_API_BASE !== 'undefined' ? window.ADMIN_API_BASE : 'UNDEFINED');
        if (typeof showError === 'function') {
            showError('API configuration error. ADMIN_API_BASE is not defined. Please refresh the page.');
        } else {
            alert('API configuration error. Please refresh the page.');
        }
        return;
    }
    
    const search = document.querySelector('[name="search"]')?.value || '';
    const status = document.querySelector('[name="status"]')?.value || 'all';
    
    showLoading('organizersTableContainer');
    
    const params = buildQueryString({ search, status });
    const apiUrl = apiBase + 'organizersapi.php' + (params ? '?' + params : '');
    console.log('Loading organizers from:', apiUrl);
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => {
        console.log('Organizers API Response status:', response.status);
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            return response.text().then(text => {
                console.error('Non-JSON response:', text.substring(0, 500));
                throw new Error('Server returned non-JSON response');
            });
        }
        if (!response.ok) {
            return response.json().then(data => {
                throw new Error(data.message || `HTTP error! status: ${response.status}`);
            });
        }
        return response.json();
    })
    .then(data => {
        console.log('Organizers API Response:', data);
        hideLoading('organizersTableContainer');
        if (data.success && data.data) {
            populateOrganizersTable(data.data.organizers || []);
        } else {
            console.error('API returned error:', data);
            showError(data.message || 'Failed to load organizers');
        }
    })
    .catch(error => {
        hideLoading('organizersTableContainer');
        console.error('Organizers API Error:', error);
        showError('Error loading organizers: ' + error.message);
    });
}

// Make function globally available
window.loadOrganizers = loadOrganizers;

function populateOrganizersTable(organizers) {
    const tbody = document.querySelector('#organizersTable tbody');
    if (!tbody) return;
    
    if (organizers.length === 0) {
        tbody.innerHTML = '<tr><td colspan="7" style="text-align: center; padding: 2rem;">No organizers found</td></tr>';
        return;
    }
    
    tbody.innerHTML = organizers.map(org => `
        <tr>
            <td>${org.id}</td>
            <td>${escapeHtml(org.company_name)}</td>
            <td>${escapeHtml(org.user_email || 'N/A')}</td>
            <td>${org.total_events || 0}</td>
            <td><span class="status-badge status-${org.kyc_status}">${capitalizeFirst(org.kyc_status)}</span></td>
            <td>${formatDate(org.created_at)}</td>
            <td>
                <a href="organizer-detail.php?id=${org.id}" class="btn-sm btn-primary">View</a>
            </td>
        </tr>
    `).join('');
}

// Set up form handler when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    const filterForm = document.getElementById('filterForm');
    if (filterForm) {
        filterForm.addEventListener('submit', function(e) {
            e.preventDefault();
            loadOrganizers();
        });
    }
});

