/**
 * Admin Notifications Page - API-based
 */

document.addEventListener('DOMContentLoaded', function() {
    console.log('Notifications page loaded');
    console.log('ADMIN_API_BASE:', typeof ADMIN_API_BASE !== 'undefined' ? ADMIN_API_BASE : 'NOT DEFINED');
    
    // Small delay to ensure admin-api.js is fully loaded
    setTimeout(function() {
        loadNotifications();
        
        const sendForm = document.getElementById('sendNotificationForm');
        if (sendForm) {
            sendForm.addEventListener('submit', function(e) {
                e.preventDefault();
                sendNotification();
            });
        }
    }, 100);
});

function loadNotifications() {
    // Function body

    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE is not defined!');
        showError('API configuration error. Please refresh the page.');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + 'notificationsapi.php';
    console.log('Loading notifications from:', apiUrl);
    showLoading('notificationsTableContainer');
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => {
        console.log('Notifications API Response status:', response.status);
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            return response.text().then(text => {
                console.error('Non-JSON response:', text.substring(0, 500));
                throw new Error('Server returned non-JSON response');
            });
        }
        if (!response.ok) {
            return response.json().then(data => {
                throw new Error(data.message || `HTTP error! status: ${response.status}`);
            });
        }
        return response.json();
    })
    .then(data => {
        console.log('Notifications API Response:', data);
        hideLoading('notificationsTableContainer');
        if (data.success && data.data) {
            populateNotificationStats(data.data.stats || {});
            populateNotificationsTable(data.data.notifications || []);
        } else {
            console.error('API returned error:', data);
            showError(data.message || 'Failed to load notifications');
        }
    })
    .catch(error => {
        hideLoading('notificationsTableContainer');
        console.error('Notifications API Error:', error);
        showError('Error loading notifications: ' + error.message);
    });
}


function sendNotification() {
    const form = document.getElementById('sendNotificationForm');
    const formData = new FormData(form);
    
    const data = {
        user_type: formData.get('user_type'),
        type: formData.get('type'),
        title: formData.get('title'),
        message: formData.get('message'),
        link: formData.get('link') || null
    };
    
    fetch(ADMIN_API_BASE + 'notificationsapi.php', {
        method: 'POST',
        credentials: 'same-origin',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify(data)
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Notification sent successfully');
            form.reset();
            loadNotifications();
        } else {
            showError(data.message || 'Failed to send notification');
        }
    })
    .catch(error => {
        console.error('Send Notification Error:', error);
        showError('Error sending notification');
    });
}

function populateNotificationStats(stats) {
    updateStatCard('total_notifications', stats.total || 0);
    updateStatCard('unread_notifications', stats.unread || 0);
}

function populateNotificationsTable(notifications) {
    const tbody = document.querySelector('#notificationsTable tbody');
    if (!tbody) return;
    
    if (notifications.length === 0) {
        tbody.innerHTML = '<tr><td colspan="6" style="text-align: center; padding: 2rem;">No notifications found</td></tr>';
        return;
    }
    
    tbody.innerHTML = notifications.map(notif => `
        <tr>
            <td>${notif.id}</td>
            <td>${capitalizeFirst(notif.user_type || 'N/A')}</td>
            <td>${escapeHtml(notif.type)}</td>
            <td>${escapeHtml(notif.title)}</td>
            <td>${notif.is_read ? '<span class="status-badge status-approved">Read</span>' : '<span class="status-badge status-pending">Unread</span>'}</td>
            <td>${formatDate(notif.created_at)}</td>
        </tr>
    `).join('');
}

function updateStatCard(statId, value) {
    const elements = document.querySelectorAll(`[data-stat="${statId}"]`);
    elements.forEach(el => {
        el.textContent = numberFormat(value);
    });
}

