/**
 * Admin Loyalty Page - API-based
 */

document.addEventListener('DOMContentLoaded', function() {
    console.log('Loyalty page loaded');
    console.log('ADMIN_API_BASE:', typeof ADMIN_API_BASE !== 'undefined' ? ADMIN_API_BASE : 'NOT DEFINED');
    
    // Small delay to ensure admin-api.js is fully loaded
    setTimeout(function() {
        loadLoyaltyData();
        
        const filterForm = document.getElementById('filterForm');
        if (filterForm) {
            filterForm.addEventListener('submit', function(e) {
                e.preventDefault();
                loadLoyaltyData();
            });
        }
    }, 100);
});

function loadLoyaltyData() {
    // Function body

    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE is not defined!');
        showError('API configuration error. Please refresh the page.');
        return;
    }
    
    const tier = document.querySelector('[name="tier"]')?.value || 'all';
    
    showLoading('loyaltyTableContainer');
    
    const params = buildQueryString({ tier });
    const apiUrl = ADMIN_API_BASE + 'loyaltyapi.php' + (params ? '?' + params : '');
    console.log('Loading loyalty data from:', apiUrl);
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => {
        console.log('Loyalty API Response status:', response.status);
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            return response.text().then(text => {
                console.error('Non-JSON response:', text.substring(0, 500));
                throw new Error('Server returned non-JSON response');
            });
        }
        if (!response.ok) {
            return response.json().then(data => {
                throw new Error(data.message || `HTTP error! status: ${response.status}`);
            });
        }
        return response.json();
    })
    .then(data => {
        console.log('Loyalty API Response:', data);
        hideLoading('loyaltyTableContainer');
        if (data.success && data.data) {
            populateLoyaltyStats(data.data.loyalty_stats || {});
            populateLoyaltyTable(data.data.loyalty_points || []);
        } else {
            console.error('API returned error:', data);
            showError(data.message || 'Failed to load loyalty data');
        }
    })
    .catch(error => {
        hideLoading('loyaltyTableContainer');
        console.error('Loyalty API Error:', error);
        showError('Error loading loyalty data: ' + error.message);
    });
}


function populateLoyaltyStats(stats) {
    updateStatCard('total_members', stats.total_members || 0);
    updateStatCard('total_points', stats.total_points || 0);
    updateStatCard('gold_members', stats.gold_members || 0);
    updateStatCard('premium_members', stats.premium_members || 0);
}

function populateLoyaltyTable(loyaltyPoints) {
    const tbody = document.querySelector('#loyaltyTable tbody');
    if (!tbody) return;
    
    if (loyaltyPoints.length === 0) {
        tbody.innerHTML = '<tr><td colspan="5" style="text-align: center; padding: 2rem;">No loyalty points found</td></tr>';
        return;
    }
    
    tbody.innerHTML = loyaltyPoints.map(lp => `
        <tr>
            <td>${escapeHtml(lp.customer_name || lp.customer_email || 'N/A')}</td>
            <td>${numberFormat(lp.points)}</td>
            <td><span class="status-badge status-${lp.tier}">${capitalizeFirst(lp.tier)}</span></td>
            <td>${formatDate(lp.updated_at)}</td>
            <td>
                <a href="customer-detail.php?id=${lp.customer_id}" class="btn-sm btn-primary">View</a>
            </td>
        </tr>
    `).join('');
}

function updateStatCard(statId, value) {
    const elements = document.querySelectorAll(`[data-stat="${statId}"]`);
    elements.forEach(el => {
        el.textContent = numberFormat(value);
    });
}

