/**
 * Universal Admin Data Loader
 * Simple, reliable API-based data loading for all admin pages
 */

(function() {
    'use strict';
    
    // Get BASE_URL from PHP or detect it automatically
    if (typeof BASE_URL === 'undefined') {
        const host = window.location.host;
        const pathname = window.location.pathname;
        
        // PRODUCTION: belive-events.videostech.cloud (no subfolder)
        if (host.includes('belive-events.videostech.cloud')) {
            window.BASE_URL = window.location.origin; // No /events_belive for production
        }
        // Alternative production domain
        else if (host.includes('recognizer.multitvsolution.com') || host.includes('multitvsolution.com')) {
            window.BASE_URL = window.location.origin + '/events';
        }
        // LOCAL/DEV: Extract /events_belive from pathname
        else {
            if (pathname.includes('/events_belive/')) {
                window.BASE_URL = window.location.origin + '/events_belive';
            } else {
                // Fallback for local development
                window.BASE_URL = window.location.origin + '/events_belive';
            }
        }
    }
    
    const API_BASE = BASE_URL + '/api/admin/';
    console.log('API Base URL:', API_BASE);
    
    // Page to API mapping
    const pageApiMap = {
        'users': 'usersapi.php',
        'organizers': 'organizersapi.php',
        'venues': 'venuesapi.php',
        'events': 'eventsapi.php',
        'wallet': 'walletapi.php',
        'loyalty': 'loyaltyapi.php',
        'analytics': 'analyticsapi.php',
        'cms': 'cmsapi.php',
        'notifications': 'notificationsapi.php',
        'settlements': 'settlementsapi.php'
    };
    
    // Detect current page
    function getCurrentPage() {
        const path = window.location.pathname;
        const match = path.match(/\/([^\/]+)\.php$/);
        if (match) {
            return match[1];
        }
        return null;
    }
    
    // Load data for current page
    function loadPageData() {
        const page = getCurrentPage();
        console.log('Current page:', page);
        
        // Settings page doesn't need API loading
        if (page === 'settings') {
            console.log('Settings page - no API loading needed');
            return;
        }
        
        if (!page || !pageApiMap[page]) {
            console.log('No API mapping for page:', page);
            return;
        }
        
        const apiUrl = API_BASE + pageApiMap[page];
        console.log('Loading data from:', apiUrl);
        
        // Analytics page has special loading (stats, not table)
        if (page === 'analytics') {
            const container = document.getElementById('analyticsContainer');
            if (container) {
                container.innerHTML = '<p>Loading analytics data...</p>';
            }
        } else {
            // Show loading in tbody only (don't replace container!)
            const tbody = document.querySelector('tbody');
            if (tbody) {
                tbody.innerHTML = '<tr><td colspan="10" style="text-align: center; padding: 2rem;">Loading data...</td></tr>';
            }
        }
        
        // Make API call
        fetch(apiUrl, {
            method: 'GET',
            credentials: 'same-origin',
            headers: {
                'Accept': 'application/json'
            }
        })
        .then(response => {
            console.log('Response status:', response.status);
            if (!response.ok) {
                throw new Error('HTTP ' + response.status);
            }
            return response.json();
        })
        .then(data => {
            console.log('API Response:', data);
            if (data.success && data.data) {
                if (page === 'analytics') {
                    renderAnalytics(data.data);
                } else {
                    renderData(page, data.data);
                }
            } else {
                showError(data.message || 'Failed to load data');
            }
        })
        .catch(error => {
            console.error('API Error:', error);
            showError('Error: ' + error.message);
        });
    }
    
    // Render analytics page (special handling for stats)
    function renderAnalytics(data) {
        console.log('Rendering analytics:', data);
        
        // Hide loading, show stats
        const container = document.getElementById('analyticsContainer');
        if (container) {
            container.style.display = 'none';
        }
        
        const statsGrid = document.getElementById('analyticsStats');
        if (statsGrid) {
            statsGrid.style.display = 'grid';
        }
        
        // Update stat cards
        if (data.analytics) {
            const stats = data.analytics;
            updateStatCard('total_events', stats.total_events || 0);
            updateStatCard('total_bookings', stats.total_bookings || 0);
            updateStatCard('total_revenue', '$' + formatCurrency(stats.total_revenue || 0));
            updateStatCard('unique_customers', stats.unique_customers || 0);
            updateStatCard('avg_booking_value', '$' + formatCurrency(stats.avg_booking_value || 0));
        }
        
        // Render revenue by month table
        if (data.revenue_by_month && data.revenue_by_month.length > 0) {
            const revenueSection = document.getElementById('revenueSection');
            if (revenueSection) {
                revenueSection.style.display = 'block';
            }
            const revenueTbody = document.querySelector('#revenueTable tbody');
            if (revenueTbody) {
                revenueTbody.innerHTML = data.revenue_by_month.map(item => `
                    <tr>
                        <td>${escapeHtml(item.month)}</td>
                        <td>$${formatCurrency(item.revenue || 0)}</td>
                        <td>${item.bookings || 0}</td>
                    </tr>
                `).join('');
            }
        }
        
        // Render events by category table
        if (data.events_by_category && data.events_by_category.length > 0) {
            const categorySection = document.getElementById('categorySection');
            if (categorySection) {
                categorySection.style.display = 'block';
            }
            const categoryTbody = document.querySelector('#categoryTable tbody');
            if (categoryTbody) {
                categoryTbody.innerHTML = data.events_by_category.map(item => `
                    <tr>
                        <td>${escapeHtml(item.category || 'Uncategorized')}</td>
                        <td>${item.event_count || 0}</td>
                        <td>$${formatCurrency(item.revenue || 0)}</td>
                    </tr>
                `).join('');
            }
        }
    }
    
    function updateStatCard(statName, value) {
        const element = document.querySelector(`[data-stat="${statName}"]`);
        if (element) {
            element.textContent = value;
        }
    }
    
    // Render data based on page
    function renderData(page, data) {
        console.log('Rendering data for:', page, data);
        
        const tbody = document.querySelector('tbody');
        if (!tbody) {
            console.error('No tbody found');
            return;
        }
        
        let items = [];
        if (page === 'users' && data.users) items = data.users;
        else if (page === 'organizers' && data.organizers) items = data.organizers;
        else if (page === 'venues' && data.venues) items = data.venues;
        else if (page === 'events' && data.events) items = data.events;
        else if (page === 'wallet' && data.transactions) items = data.transactions;
        else if (page === 'loyalty' && data.loyalty_points) items = data.loyalty_points;
        else if (page === 'cms' && data.pages) items = data.pages;
        else if (page === 'notifications' && data.notifications) items = data.notifications;
        else if (page === 'settlements' && data.organizer_settlements) items = data.organizer_settlements;
        else items = Object.values(data)[0] || [];
        
        if (items.length === 0) {
            tbody.innerHTML = '<tr><td colspan="10" style="text-align: center; padding: 2rem;">No data found</td></tr>';
            return;
        }
        
        // Render based on page type
        if (page === 'users') {
            tbody.innerHTML = items.map(item => `
                <tr>
                    <td>${item.id}</td>
                    <td>${escapeHtml(item.full_name || (item.first_name + ' ' + item.last_name) || 'N/A')}</td>
                    <td>${escapeHtml(item.email)}</td>
                    <td>${item.provider || 'email'}</td>
                    <td>${item.total_bookings || 0}</td>
                    <td><span class="status-badge ${item.is_active ? 'status-approved' : 'status-rejected'}">${item.is_active ? 'Active' : 'Inactive'}</span></td>
                    <td>${formatDate(item.created_at)}</td>
                    <td><a href="#" class="btn-sm btn-primary">View</a></td>
                </tr>
            `).join('');
        } else if (page === 'organizers') {
            tbody.innerHTML = items.map(item => `
                <tr>
                    <td>${item.id}</td>
                    <td>${escapeHtml(item.company_name)}</td>
                    <td>${escapeHtml(item.contact_email || item.user_email || 'N/A')}</td>
                    <td>${escapeHtml(item.user_name || 'N/A')}</td>
                    <td>${item.total_events || 0}</td>
                    <td><span class="status-badge status-${item.kyc_status}">${capitalizeFirst(item.kyc_status)}</span></td>
                    <td><span class="status-badge ${item.is_active ? 'status-approved' : 'status-rejected'}">${item.is_active ? 'Active' : 'Inactive'}</span></td>
                    <td>${formatDate(item.created_at)}</td>
                    <td><a href="#" class="btn-sm btn-primary">View</a></td>
                </tr>
            `).join('');
        } else if (page === 'venues') {
            tbody.innerHTML = items.map(item => `
                <tr>
                    <td>${item.id}</td>
                    <td>${escapeHtml(item.venue_name)}</td>
                    <td>${escapeHtml((item.city || '') + ', ' + (item.country || ''))}</td>
                    <td>${item.capacity || 'N/A'}</td>
                    <td>${item.total_events || 0}</td>
                    <td><span class="status-badge status-${item.verification_status}">${capitalizeFirst(item.verification_status)}</span></td>
                    <td>${formatDate(item.created_at)}</td>
                    <td><a href="#" class="btn-sm btn-primary">View</a></td>
                </tr>
            `).join('');
        } else if (page === 'events') {
            tbody.innerHTML = items.map(item => `
                <tr>
                    <td>${item.id}</td>
                    <td>${escapeHtml(item.name)}</td>
                    <td>${escapeHtml(item.organizer_name || 'N/A')}</td>
                    <td>${escapeHtml(item.venue_name || 'N/A')}</td>
                    <td>${formatDate(item.start_date)}</td>
                    <td><span class="status-badge status-${item.status}">${capitalizeFirst(item.status)}</span></td>
                    <td>${formatDate(item.created_at)}</td>
                    <td><a href="#" class="btn-sm btn-primary">View</a></td>
                </tr>
            `).join('');
        } else if (page === 'wallet') {
            // Update wallet stats if available
            if (data.wallet_stats) {
                updateStatCard('total_wallets', data.wallet_stats.total_wallets || 0);
                updateStatCard('total_balance', '$' + formatCurrency(data.wallet_stats.total_balance || 0));
                updateStatCard('active_wallets', data.wallet_stats.active_wallets || 0);
            }
            tbody.innerHTML = items.map(item => `
                <tr>
                    <td>${item.id}</td>
                    <td>${escapeHtml(item.customer_name || item.customer_email || 'N/A')}</td>
                    <td><span class="status-badge ${item.transaction_type === 'credit' ? 'status-approved' : 'status-rejected'}">${capitalizeFirst(item.transaction_type)}</span></td>
                    <td>$${formatCurrency(item.amount)}</td>
                    <td>$${formatCurrency(item.balance_after)}</td>
                    <td>${escapeHtml(item.description || 'N/A')}</td>
                    <td>${formatDate(item.created_at)}</td>
                </tr>
            `).join('');
        } else if (page === 'loyalty') {
            // Update loyalty stats if available
            if (data.loyalty_stats) {
                updateStatCard('total_members', data.loyalty_stats.total_members || 0);
                updateStatCard('total_points', data.loyalty_stats.total_points || 0);
                updateStatCard('gold_members', data.loyalty_stats.gold_members || 0);
                updateStatCard('premium_members', data.loyalty_stats.premium_members || 0);
            }
            tbody.innerHTML = items.map(item => `
                <tr>
                    <td>${escapeHtml(item.customer_name || item.customer_email || 'N/A')}</td>
                    <td>${item.points || 0}</td>
                    <td><span class="status-badge status-${item.tier}">${capitalizeFirst(item.tier)}</span></td>
                    <td>${formatDate(item.updated_at || item.created_at)}</td>
                    <td><a href="#" class="btn-sm btn-primary">View</a></td>
                </tr>
            `).join('');
        } else if (page === 'cms') {
            tbody.innerHTML = items.map(item => `
                <tr>
                    <td>${escapeHtml(item.page_slug)}</td>
                    <td>${escapeHtml(item.page_title)}</td>
                    <td><span class="status-badge ${item.is_active ? 'status-approved' : 'status-rejected'}">${item.is_active ? 'Active' : 'Inactive'}</span></td>
                    <td>${formatDate(item.updated_at || item.created_at)}</td>
                    <td><a href="#" class="btn-sm btn-primary">Edit</a></td>
                </tr>
            `).join('');
        } else if (page === 'notifications') {
            // Update notification stats if available
            if (data.stats) {
                updateStatCard('total_notifications', data.stats.total || 0);
                updateStatCard('unread_notifications', data.stats.unread || 0);
            }
            tbody.innerHTML = items.map(item => `
                <tr>
                    <td>${item.id}</td>
                    <td>${capitalizeFirst(item.user_type || 'all')}</td>
                    <td>${escapeHtml(item.type)}</td>
                    <td>${escapeHtml(item.title)}</td>
                    <td><span class="status-badge ${item.is_read ? 'status-approved' : 'status-pending'}">${item.is_read ? 'Read' : 'Unread'}</span></td>
                    <td>${formatDate(item.created_at)}</td>
                </tr>
            `).join('');
        } else if (page === 'settlements') {
            // Update settlement stats if available
            if (data.settlement_stats) {
                const container = document.getElementById('settlementsContainer');
                if (container) container.style.display = 'none';
                const statsGrid = document.getElementById('settlementStats');
                if (statsGrid) statsGrid.style.display = 'grid';
                const tableSection = document.getElementById('settlementsTableSection');
                if (tableSection) tableSection.style.display = 'block';
                
                updateStatCard('total_organizers', data.settlement_stats.total_organizers || 0);
                updateStatCard('total_revenue', '$' + formatCurrency(data.settlement_stats.total_revenue || 0));
                updateStatCard('platform_commission', '$' + formatCurrency(data.settlement_stats.platform_commission || 0));
                updateStatCard('organizer_payout', '$' + formatCurrency(data.settlement_stats.organizer_payout || 0));
            }
            tbody.innerHTML = items.map(item => `
                <tr>
                    <td>${escapeHtml(item.company_name)}</td>
                    <td>$${formatCurrency(item.total_revenue)}</td>
                    <td>$${formatCurrency(item.commission)}</td>
                    <td>$${formatCurrency(item.payout)}</td>
                    <td>${item.total_bookings || 0}</td>
                    <td><a href="#" class="btn-sm btn-primary">View Details</a></td>
                </tr>
            `).join('');
        } else {
            // Generic render
            tbody.innerHTML = items.map((item, idx) => {
                const keys = Object.keys(item).slice(0, 8);
                return `<tr>${keys.map(key => `<td>${escapeHtml(String(item[key] || ''))}</td>`).join('')}</tr>`;
            }).join('');
        }
    }
    
    // Helper functions
    function escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    function formatDate(dateString) {
        if (!dateString) return 'N/A';
        try {
            const date = new Date(dateString);
            return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
        } catch (e) {
            return dateString;
        }
    }
    
    function formatCurrency(amount) {
        if (!amount && amount !== 0) return '0.00';
        return parseFloat(amount).toFixed(2);
    }
    
    function capitalizeFirst(str) {
        if (!str) return '';
        return str.charAt(0).toUpperCase() + str.slice(1);
    }
    
    function showError(message) {
        const tbody = document.querySelector('tbody');
        if (tbody) {
            tbody.innerHTML = '<tr><td colspan="10" style="text-align: center; padding: 2rem; color: #dc2626;">' + escapeHtml(message) + '</td></tr>';
        } else {
            const container = document.querySelector('.admin-content, .table-container');
            if (container) {
                container.innerHTML = '<div style="background: #fee2e2; color: #dc2626; padding: 1rem; border-radius: 8px;">' + escapeHtml(message) + '</div>';
            }
        }
    }
    
    // Start loading when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', loadPageData);
    } else {
        loadPageData();
    }
    
    // Also try after a delay
    setTimeout(loadPageData, 500);
    
})();
