/**
 * Universal Admin Page Initialization
 * This script ensures API calls are made after all scripts are loaded
 */

(function() {
    'use strict';
    
    console.log('🔧 admin-init.js loaded and executing...');
    console.log('Current URL:', window.location.href);
    console.log('Current pathname:', window.location.pathname);
    
    // Map page names to their load functions
    const pageLoadFunctions = {
        'users': 'loadUsers',
        'events': 'loadEvents',
        'organizers': 'loadOrganizers',
        'venues': 'loadVenues',
        'wallet': 'loadWalletData',
        'loyalty': 'loadLoyaltyData',
        'analytics': 'loadAnalytics',
        'cms': 'loadCMSData',
        'notifications': 'loadNotifications',
        'settlements': 'loadSettlements'
    };
    
    function detectPageName() {
        // Try to get page name from URL
        const path = window.location.pathname;
        console.log('Detecting page from path:', path);
        
        // First, check if path contains /pages/
        if (path.includes('/pages/')) {
            const parts = path.split('/pages/');
            if (parts[1]) {
                const pageName = parts[1].replace('.php', '').split('?')[0];
                console.log('Detected from /pages/ path:', pageName);
                if (pageLoadFunctions[pageName]) {
                    return pageName;
                }
            }
        }
        
        // Try to match filename at end of path
        const match = path.match(/\/([^\/]+)\.php/);
        if (match && match[1]) {
            const pageName = match[1].split('?')[0];
            console.log('Detected from filename:', pageName);
            if (pageLoadFunctions[pageName]) {
                return pageName;
            }
        }
        
        // Try to get from body class or data attribute
        const body = document.body;
        if (body) {
            for (const page in pageLoadFunctions) {
                if (body.classList.contains('page-' + page)) {
                    console.log('Detected from body class:', page);
                    return page;
                }
            }
        }
        
        console.warn('Could not detect page name from path:', path);
        return null;
    }
    
    function initPage() {
        const pageName = detectPageName();
        console.log('=== Admin Page Initialization ===');
        console.log('Detected page:', pageName);
        console.log('BASE_URL:', typeof BASE_URL !== 'undefined' ? BASE_URL : 'UNDEFINED');
        console.log('ADMIN_API_BASE:', typeof ADMIN_API_BASE !== 'undefined' ? ADMIN_API_BASE : 'UNDEFINED');
        
        if (!pageName || !pageLoadFunctions[pageName]) {
            console.log('No load function for this page or page not detected');
            return;
        }
        
        const loadFunctionName = pageLoadFunctions[pageName];
        console.log('Load function:', loadFunctionName);
        console.log('Function exists:', typeof window[loadFunctionName] !== 'undefined' ? 'YES' : 'NO');
        
        if (typeof ADMIN_API_BASE === 'undefined') {
            console.error('❌ ADMIN_API_BASE is not defined!');
            console.error('Scripts may not have loaded correctly. Retrying in 500ms...');
            setTimeout(initPage, 500);
            return;
        }
        
        if (typeof window[loadFunctionName] === 'function') {
            console.log('✅ Calling', loadFunctionName, '...');
            try {
                window[loadFunctionName]();
                console.log('✅', loadFunctionName, 'called successfully');
            } catch (error) {
                console.error('❌ Error calling', loadFunctionName, ':', error);
                console.error('Error stack:', error.stack);
            }
        } else {
            console.error('❌ Function', loadFunctionName, 'is not defined!');
            console.error('Available functions:', Object.keys(window).filter(k => k.startsWith('load')));
            console.error('Checking direct access:', typeof loadOrganizers, typeof loadUsers, typeof loadEvents);
            // Retry after a delay
            setTimeout(initPage, 500);
        }
    }
    
    // Multiple initialization strategies
    function startInit() {
        console.log('🚀 Starting admin page initialization...');
        console.log('Document readyState:', document.readyState);
        
        // Strategy 1: Immediate execution if scripts are already loaded
        function tryInit() {
            const pageName = detectPageName();
            if (!pageName) {
                console.warn('Page name not detected yet, will retry...');
                return false;
            }
            
            const loadFunctionName = pageLoadFunctions[pageName];
            if (typeof ADMIN_API_BASE === 'undefined') {
                console.warn('ADMIN_API_BASE not ready yet, will retry...');
                return false;
            }
            
            if (typeof window[loadFunctionName] !== 'function') {
                console.warn('Load function', loadFunctionName, 'not ready yet, will retry...');
                return false;
            }
            
            console.log('✅ All conditions met, calling', loadFunctionName);
            try {
                window[loadFunctionName]();
                return true;
            } catch (error) {
                console.error('❌ Error calling', loadFunctionName, ':', error);
                return false;
            }
        }
        
        // Try immediately
        if (!tryInit()) {
            // Strategy 2: Wait for DOMContentLoaded
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', function() {
                    console.log('📄 DOMContentLoaded fired');
                    setTimeout(function() {
                        if (!tryInit()) {
                            setTimeout(tryInit, 500);
                        }
                    }, 100);
                });
            } else {
                // DOM already loaded
                setTimeout(function() {
                    if (!tryInit()) {
                        setTimeout(tryInit, 500);
                    }
                }, 100);
            }
            
            // Strategy 3: Fallback after window load
            window.addEventListener('load', function() {
                console.log('🪟 Window load event fired');
                setTimeout(function() {
                    if (!tryInit()) {
                        console.error('❌ Failed to initialize after all retries');
                    }
                }, 200);
            });
        }
    }
    
    // Start initialization immediately
    startInit();
})();

