/**
 * Admin Events Page - API-based
 */

document.addEventListener('DOMContentLoaded', function() {
    console.log('Events page loaded');
    console.log('ADMIN_API_BASE:', typeof ADMIN_API_BASE !== 'undefined' ? ADMIN_API_BASE : 'NOT DEFINED');
    
    // Small delay to ensure admin-api.js is fully loaded
    setTimeout(function() {
        loadEvents();
        
        const filterForm = document.getElementById('filterForm');
        if (filterForm) {
            filterForm.addEventListener('submit', function(e) {
                e.preventDefault();
                loadEvents();
            });
        }
    }, 100);
});

function loadEvents() {
    // Function body starts here
    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE is not defined!');
        showError('API configuration error. Please refresh the page.');
        return;
    }
    
    const search = document.querySelector('[name="search"]')?.value || '';
    const status = document.querySelector('[name="status"]')?.value || 'all';
    
    showLoading('eventsTableContainer');
    
    const params = buildQueryString({ search, status });
    const apiUrl = ADMIN_API_BASE + 'eventsapi.php' + (params ? '?' + params : '');
    console.log('Loading events from:', apiUrl);
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => {
        console.log('Events API Response status:', response.status);
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        console.log('Events API Response:', data);
        hideLoading('eventsTableContainer');
        if (data.success && data.data) {
            populateEventsTable(data.data.events || []);
        } else {
            console.error('API returned error:', data);
            showError(data.message || 'Failed to load events');
        }
    })
    .catch(error => {
        hideLoading('eventsTableContainer');
        console.error('Events API Error:', error);
        showError('Error loading events: ' + error.message);
    });
}

// Make function globally available
window.loadEvents = loadEvents;

// Set up form handler when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    const filterForm = document.getElementById('filterForm');
    if (filterForm) {
        filterForm.addEventListener('submit', function(e) {
            e.preventDefault();
            loadEvents();
        });
    }
});

function populateEventsTable(events) {
    const tbody = document.querySelector('#eventsTable tbody');
    if (!tbody) return;
    
    if (events.length === 0) {
        tbody.innerHTML = '<tr><td colspan="7" style="text-align: center; padding: 2rem;">No events found</td></tr>';
        return;
    }
    
    tbody.innerHTML = events.map(event => `
        <tr>
            <td>${event.id}</td>
            <td>${escapeHtml(event.name || 'Untitled')}</td>
            <td>${escapeHtml(event.organizer_name || 'N/A')}</td>
            <td>${escapeHtml(event.venue_name || 'N/A')}</td>
            <td><span class="status-badge status-${event.status}">${capitalizeFirst(event.status)}</span></td>
            <td>${formatDate(event.start_date)}</td>
            <td>
                <a href="event-detail.php?id=${event.id}" class="btn-sm btn-primary">View</a>
            </td>
        </tr>
    `).join('');
}

