/**
 * Admin Dashboard - API-based data loading
 */

// Wait for admin-api.js to load first
// Note: This will be called after admin-api.js loads
// The actual call is handled in dashboard.php to ensure proper script loading order

function loadDashboardData() {
    console.log('=== loadDashboardData called ===');
    
    // Check for ADMIN_API_BASE in both global scope and window object
    const apiBase = typeof ADMIN_API_BASE !== 'undefined' ? ADMIN_API_BASE : (typeof window.ADMIN_API_BASE !== 'undefined' ? window.ADMIN_API_BASE : undefined);
    
    console.log('ADMIN_API_BASE:', apiBase || 'UNDEFINED');
    console.log('window.ADMIN_API_BASE:', typeof window.ADMIN_API_BASE !== 'undefined' ? window.ADMIN_API_BASE : 'UNDEFINED');
    
    if (!apiBase) {
        console.error('❌ ADMIN_API_BASE is not defined!');
        console.error('BASE_URL:', typeof BASE_URL !== 'undefined' ? BASE_URL : (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : 'UNDEFINED'));
        if (typeof showError === 'function') {
            showError('API configuration error. Please refresh the page.');
        } else {
            alert('API configuration error. Please refresh the page.');
        }
        return;
    }
    
    const apiUrl = apiBase + 'dashboardapi.php';
    console.log('🚀 Making API call to:', apiUrl);
    console.log('Full URL:', window.location.origin + window.location.pathname.split('/').slice(0, -2).join('/') + '/api/admin/dashboardapi.php');
    
    if (typeof showLoading === 'function') {
        showLoading('loadingState');
    } else {
        console.warn('showLoading function not available');
    }
    
    console.log('📡 Fetch request starting...');
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => {
        console.log('Response status:', response.status);
        console.log('Response Content-Type:', response.headers.get('content-type'));
        
        // Check if response is JSON
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            return response.text().then(text => {
                console.error('Non-JSON response received:', text.substring(0, 500));
                throw new Error('Server returned non-JSON response. Status: ' + response.status + '. Response: ' + text.substring(0, 200));
            });
        }
        
        if (!response.ok) {
            return response.json().then(data => {
                console.error('API Error Response:', data);
                throw new Error(data.message || data.error || `HTTP error! status: ${response.status}`);
            });
        }
        return response.json();
    })
    .then(data => {
        console.log('Dashboard API Response:', data);
        console.log('Response structure:', {
            success: data.success,
            hasData: !!data.data,
            recentEvents: data.data?.recent_events?.length || 0
        });
        hideLoading('loadingState');
        const statsGrid = document.getElementById('statsGrid');
        if (statsGrid) {
            statsGrid.style.display = 'grid';
        }
        if (data.success && data.data) {
            console.log('Calling populateDashboard with:', data.data);
            populateDashboard(data.data);
        } else {
            console.error('API returned error:', data);
            const errorMsg = data.message || 'Failed to load dashboard data';
            console.error('Error message:', errorMsg);
            showError(errorMsg);
            // Show error in table too
            const tbody = document.querySelector('#recentEventsTable tbody');
            if (tbody) {
                tbody.innerHTML = '<tr><td colspan="6" style="text-align: center; padding: 2rem; color: #dc2626;">' + errorMsg + '</td></tr>';
            }
        }
    })
    .catch(error => {
        hideLoading('loadingState');
        console.error('Dashboard API Error:', error);
        showError('Error loading dashboard data: ' + error.message);
    });
}

function populateDashboard(data) {
    console.log('=== populateDashboard called ===');
    console.log('Full data object:', data);
    const stats = data.stats || {};
    const performanceData = data.performance_data || [];
    const statusDistribution = data.status_distribution || [];
    const recentEvents = data.recent_events || [];
    
    console.log('Stats to populate:', stats);
    console.log('Performance data:', performanceData);
    console.log('Status distribution:', statusDistribution);
    console.log('Recent events array:', recentEvents);
    console.log('Recent events count:', recentEvents.length);
    
    // Update summary cards
    updateSummaryCard('totalEvents', stats.total_events || 0);
    updateSummaryCard('publishedEvents', stats.published_events || 0);
    updateSummaryCard('draftEvents', stats.draft_events || 0);
    updateSummaryCard('cancelledEvents', stats.cancelled_events || 0);
    
    // Update percentage changes (mock data for now)
    updatePercentageChange('totalEventsChange', 0.29);
    updatePercentageChange('publishedEventsChange', 0.29);
    updatePercentageChange('draftEventsChange', 0.18);
    updatePercentageChange('cancelledEventsChange', -0.12);
    
    // Update tab counts
    updateTabCount('tabEventsCount', stats.total_events || 0);
    updateTabCount('tabUsersCount', stats.total_users || 0);
    updateTabCount('tabOrganizersCount', stats.total_organizers || 0);
    updateTabCount('tabVenuesCount', stats.total_venues || 0);
    
    // Initialize charts
    initPerformanceChart(performanceData);
    initStatusChart(statusDistribution);
    
    // Populate recent events table (if section exists)
    if (document.getElementById('recentEventsSection')) {
        populateRecentEvents(recentEvents);
    }
}

function updateSummaryCard(elementId, value) {
    const element = document.getElementById(elementId);
    if (element) {
        element.textContent = getNumberFormat(value);
    }
}

function updatePercentageChange(elementId, percentage) {
    const element = document.getElementById(elementId);
    if (element) {
        const percent = Math.abs(percentage * 100).toFixed(2);
        element.textContent = percent + '%';
    }
}

function updateTabCount(elementId, value) {
    const element = document.getElementById(elementId);
    if (element) {
        element.textContent = getNumberFormat(value);
    }
}

function initPerformanceChart(data) {
    const ctx = document.getElementById('performanceChart');
    if (!ctx) return;
    
    const labels = data.map(d => d.date);
    const counts = data.map(d => d.count);
    
    if (window.performanceChartInstance) {
        window.performanceChartInstance.destroy();
    }
    
    window.performanceChartInstance = new Chart(ctx, {
        type: 'line',
        data: {
            labels: labels,
            datasets: [{
                label: 'Events Created',
                data: counts,
                borderColor: '#dc2626',
                backgroundColor: 'rgba(220, 38, 38, 0.1)',
                tension: 0.4,
                fill: true
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        stepSize: 1
                    }
                }
            }
        }
    });
}

function initStatusChart(data) {
    const ctx = document.getElementById('statusChart');
    if (!ctx) return;
    
    const labels = data.map(d => d.label);
    const values = data.map(d => d.value);
    const colors = data.map(d => d.color);
    
    if (window.statusChartInstance) {
        window.statusChartInstance.destroy();
    }
    
    window.statusChartInstance = new Chart(ctx, {
        type: 'doughnut',
        data: {
            labels: labels,
            datasets: [{
                data: values,
                backgroundColor: colors,
                borderWidth: 0
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom',
                    labels: {
                        padding: 15,
                        usePointStyle: true
                    }
                }
            }
        }
    });
}

// Old updateStatCard function removed - using updateSummaryCard instead

function populateRecentEvents(events) {
    console.log('=== populateRecentEvents called ===');
    console.log('Events received:', events);
    console.log('Events count:', events ? events.length : 0);
    
    // Ensure section is visible
    const section = document.getElementById('recentEventsSection');
    if (section) {
        section.style.display = 'block';
    }
    
    const tbody = document.querySelector('#recentEventsTable tbody');
    if (!tbody) {
        console.error('❌ Table body (#recentEventsTable tbody) not found!');
        console.error('Available tables:', document.querySelectorAll('table').length);
        return;
    }
    
    console.log('✅ Table body found');
    
    if (!events || events.length === 0) {
        console.warn('⚠️ No events to display');
        tbody.innerHTML = '<tr><td colspan="6" style="text-align: center; padding: 2rem;">No events found</td></tr>';
        return;
    }
    
    try {
        console.log('Processing', events.length, 'events for display');
        const rows = events.map((event, index) => {
            // console.log(`Processing event ${index + 1}:`, {
            //     id: event.id,
            //     name: event.name,
            //     status: event.status,
            //     organizer: event.organizer_name
            // });
            
            const status = (event.status || 'pending').toLowerCase();
            const statusClass = `status-${status}`;
            return `
                <tr>
                    <td>${event.id || 'N/A'}</td>
                    <td>${getEscapeHtml(event.name || 'Untitled Event')}</td>
                    <td>${getEscapeHtml(event.organizer_name || 'N/A')}</td>
                    <td><span class="status-badge ${statusClass}">${getCapitalizeFirst(event.status || 'pending')}</span></td>
                    <td>${getFormatDate(event.start_date)}</td>
                    <td>
                        <a href="pages/events.php" class="btn-sm btn-primary">View</a>
                    </td>
                </tr>
            `;
        }).join('');
        
        tbody.innerHTML = rows;
        console.log('✅ Recent events table populated successfully with', events.length, 'events');
        console.log('Table HTML length:', rows.length, 'characters');
    } catch (error) {
        console.error('❌ Error populating events table:', error);
        console.error('Error stack:', error.stack);
        tbody.innerHTML = '<tr><td colspan="6" style="text-align: center; padding: 2rem; color: #dc2626;">Error displaying events: ' + error.message + '</td></tr>';
    }
}

// Helper functions - use from admin-api.js if available, otherwise define fallbacks
// These functions are already defined in admin-api.js and available on window object
// We'll use them directly from window to avoid recursion issues

// Helper to safely get numberFormat
function getNumberFormat(num, decimals = 0) {
    if (typeof window.numberFormat === 'function') {
        return window.numberFormat(num, decimals);
    }
    // Fallback
    return new Intl.NumberFormat('en-US', {
        minimumFractionDigits: decimals,
        maximumFractionDigits: decimals
    }).format(num);
}

// Helper to safely get formatDate
function getFormatDate(dateString) {
    if (typeof window.formatDate === 'function') {
        return window.formatDate(dateString);
    }
    // Fallback
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
}

// Helper to safely get capitalizeFirst
function getCapitalizeFirst(str) {
    if (typeof window.capitalizeFirst === 'function') {
        return window.capitalizeFirst(str);
    }
    // Fallback
    if (!str) return '';
    return str.charAt(0).toUpperCase() + str.slice(1).toLowerCase();
}

// Helper to safely get escapeHtml
function getEscapeHtml(text) {
    if (typeof window.escapeHtml === 'function') {
        return window.escapeHtml(text);
    }
    // Fallback
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// showError is now provided by admin-api.js globally
// This local definition is kept as fallback only
if (typeof window.showError === 'undefined') {
    function showError(message) {
        const errorDiv = document.createElement('div');
        errorDiv.className = 'error-message';
        errorDiv.style.cssText = 'background: #fee2e2; color: #dc2626; padding: 1rem; border-radius: 8px; margin-bottom: 1rem;';
        errorDiv.textContent = message;
        const content = document.querySelector('.admin-content');
        if (content) {
            content.insertBefore(errorDiv, content.firstChild);
        }
    }
    window.showError = showError;
}

// Make function globally available
window.loadDashboardData = loadDashboardData;

