/**
 * Admin CMS Page - API-based
 */

document.addEventListener('DOMContentLoaded', function() {
    console.log('CMS page loaded');
    console.log('ADMIN_API_BASE:', typeof ADMIN_API_BASE !== 'undefined' ? ADMIN_API_BASE : 'NOT DEFINED');
    
    // Small delay to ensure admin-api.js is fully loaded
    setTimeout(function() {
        loadCMSData();
    }, 100);
});

function loadCMSData() {
    // Function body

    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE is not defined!');
        showError('API configuration error. Please refresh the page.');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + 'cmsapi.php';
    console.log('Loading CMS data from:', apiUrl);
    showLoading('cmsContainer');
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => {
        console.log('CMS API Response status:', response.status);
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            return response.text().then(text => {
                console.error('Non-JSON response:', text.substring(0, 500));
                throw new Error('Server returned non-JSON response');
            });
        }
        if (!response.ok) {
            return response.json().then(data => {
                throw new Error(data.message || `HTTP error! status: ${response.status}`);
            });
        }
        return response.json();
    })
    .then(data => {
        console.log('CMS API Response:', data);
        hideLoading('cmsContainer');
        const pagesEl = document.getElementById('pagesSection');
        const bannersEl = document.getElementById('bannersSection');
        if (pagesEl) pagesEl.style.display = 'block';
        if (bannersEl) bannersEl.style.display = 'block';
        if (data.success && data.data) {
            populateCMSPages(data.data.pages || []);
            populateBanners(data.data.banners || []);
        } else {
            console.error('API returned error:', data);
            showError(data.message || 'Failed to load CMS data');
        }
    })
    .catch(error => {
        hideLoading('cmsContainer');
        console.error('CMS API Error:', error);
        showError('Error loading CMS data: ' + error.message);
    });
}


function populateCMSPages(pages) {
    const tbody = document.querySelector('#pagesTable tbody');
    if (!tbody) return;
    
    if (pages.length === 0) {
        tbody.innerHTML = '<tr><td colspan="4" style="text-align: center; padding: 2rem;">No CMS pages found</td></tr>';
        return;
    }
    
    tbody.innerHTML = pages.map(page => `
        <tr>
            <td>${escapeHtml(page.page_slug || 'N/A')}</td>
            <td>${escapeHtml(page.title || 'N/A')}</td>
            <td>${formatDate(page.updated_at)}</td>
            <td>
                <a href="edit-page.php?id=${page.id}" class="btn-sm btn-primary">Edit</a>
            </td>
        </tr>
    `).join('');
}

function populateBanners(banners) {
    const tbody = document.querySelector('#bannersTable tbody');
    if (!tbody) return;
    
    if (banners.length === 0) {
        tbody.innerHTML = '<tr><td colspan="5" style="text-align: center; padding: 2rem;">No banners found</td></tr>';
        return;
    }
    
    tbody.innerHTML = banners.map(banner => `
        <tr>
            <td>${escapeHtml(banner.title || 'N/A')}</td>
            <td>${escapeHtml(banner.position || 'N/A')}</td>
            <td>${banner.is_active ? '<span class="status-badge status-approved">Active</span>' : '<span class="status-badge status-rejected">Inactive</span>'}</td>
            <td>${formatDate(banner.created_at)}</td>
            <td>
                <a href="edit-banner.php?id=${banner.id}" class="btn-sm btn-primary">Edit</a>
            </td>
        </tr>
    `).join('');
}

