/**
 * Admin Panel - Shared API Functions
 */

// Base API URL - ensure BASE_URL is defined
console.log('🔧 admin-api.js loading...');
console.log('🔧 Checking BASE_URL:', typeof BASE_URL !== 'undefined' ? BASE_URL : 'UNDEFINED');
console.log('🔧 Checking window.BASE_URL:', typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : 'UNDEFINED');

// Use existing BASE_URL if defined, otherwise detect it
let baseUrl;
if (typeof BASE_URL !== 'undefined' && BASE_URL) {
    baseUrl = BASE_URL;
    console.log('✅ Using existing BASE_URL:', baseUrl);
} else if (typeof window.BASE_URL !== 'undefined' && window.BASE_URL) {
    baseUrl = window.BASE_URL;
    console.log('✅ Using window.BASE_URL:', baseUrl);
} else {
    console.warn('⚠️ BASE_URL is not defined, attempting to detect...');
    // Auto-detect BASE_URL from current URL
    const host = window.location.host;
    const pathname = window.location.pathname;
    
    // PRODUCTION: belive-events.videostech.cloud (no subfolder)
    if (host.includes('belive-events.videostech.cloud')) {
        baseUrl = window.location.origin; // No /events_belive for production
    }
    // Alternative production domain
    else if (host.includes('recognizer.multitvsolution.com') || host.includes('multitvsolution.com')) {
        baseUrl = window.location.origin + '/events';
    }
    // LOCAL/DEV: Extract /events_belive from pathname
    else {
        // Check if pathname contains /events_belive
        if (pathname.includes('/events_belive/')) {
            baseUrl = window.location.origin + '/events_belive';
        } else {
            // Try to extract from path structure
            const pathParts = pathname.split('/').filter(p => p);
            const eventsIndex = pathParts.indexOf('events_belive');
            if (eventsIndex !== -1) {
                baseUrl = window.location.origin + '/' + pathParts.slice(0, eventsIndex + 1).join('/');
            } else {
                // Fallback for local development
                baseUrl = window.location.origin + '/events_belive';
            }
        }
    }
    
    console.log('✅ Detected BASE_URL:', baseUrl);
    // Set it on window object for global access
    window.BASE_URL = baseUrl;
}

// Verify baseUrl is set before creating ADMIN_API_BASE
if (!baseUrl) {
    console.error('❌ CRITICAL ERROR: BASE_URL is still undefined after detection!');
    // Final fallback based on host
    const host = window.location.host;
    if (host.includes('belive-events.videostech.cloud')) {
        baseUrl = window.location.origin;
    } else {
        baseUrl = window.location.origin + '/events_belive';
    }
    window.BASE_URL = baseUrl;
    console.warn('⚠️ Using fallback BASE_URL:', baseUrl);
}

// Base API URL
const ADMIN_API_BASE = baseUrl + '/api/admin/';
console.log('✅ ADMIN_API_BASE initialized:', ADMIN_API_BASE);
console.log('✅ Full API URL example:', ADMIN_API_BASE + 'organizersapi.php');

// Make ADMIN_API_BASE globally available
window.ADMIN_API_BASE = ADMIN_API_BASE;

/**
 * Generic API fetch function
 */
async function fetchAdminAPI(endpoint, options = {}) {
    const url = ADMIN_API_BASE + endpoint;
    const defaultOptions = {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Content-Type': 'application/json'
        }
    };
    
    const config = { ...defaultOptions, ...options };
    
    try {
        const response = await fetch(url, config);
        const data = await response.json();
        
        if (!response.ok) {
            throw new Error(data.message || 'API request failed');
        }
        
        return data;
    } catch (error) {
        console.error('API Error:', error);
        throw error;
    }
}

/**
 * Format number with commas
 */
function numberFormat(num, decimals = 0) {
    const formatted = new Intl.NumberFormat('en-US', {
        minimumFractionDigits: decimals,
        maximumFractionDigits: decimals
    }).format(num);
    return formatted;
}
// Make available globally
window.numberFormat = numberFormat;

/**
 * Format date
 */
function formatDate(dateString) {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
}
// Make available globally
window.formatDate = formatDate;

/**
 * Format currency
 */
function formatCurrency(amount) {
    return '$' + numberFormat(amount, 2);
}
// Make available globally
window.formatCurrency = formatCurrency;

/**
 * Capitalize first letter
 */
function capitalizeFirst(str) {
    if (!str) return '';
    return str.charAt(0).toUpperCase() + str.slice(1).toLowerCase();
}
// Make available globally
window.capitalizeFirst = capitalizeFirst;

/**
 * Escape HTML
 */
function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
// Make available globally
window.escapeHtml = escapeHtml;

/**
 * Show error message
 */
function showError(message, container = '.admin-content') {
    const errorDiv = document.createElement('div');
    errorDiv.className = 'error-message';
    errorDiv.style.cssText = 'background: #fee2e2; color: #dc2626; padding: 1rem; border-radius: 8px; margin-bottom: 1rem;';
    errorDiv.textContent = message;
    const content = document.querySelector(container);
    if (content) {
        content.insertBefore(errorDiv, content.firstChild);
        setTimeout(() => errorDiv.remove(), 5000);
    }
}
// Make available globally
window.showError = showError;

/**
 * Show loading state
 */
function showLoading(elementId) {
    const element = document.getElementById(elementId);
    if (element) {
        element.style.display = 'block';
        element.innerHTML = '<p style="text-align: center; padding: 2rem;">Loading...</p>';
    } else {
        console.warn('showLoading: Element not found:', elementId);
    }
}
// Make available globally
window.showLoading = showLoading;

/**
 * Hide loading state
 */
function hideLoading(elementId) {
    const element = document.getElementById(elementId);
    if (element) {
        element.style.display = 'none';
    } else {
        console.warn('hideLoading: Element not found:', elementId);
    }
}
// Make available globally
window.hideLoading = hideLoading;

/**
 * Build query string from object
 */
function buildQueryString(params) {
    const query = new URLSearchParams();
    Object.keys(params).forEach(key => {
        if (params[key] !== null && params[key] !== undefined && params[key] !== '') {
            query.append(key, params[key]);
        }
    });
    return query.toString();
}
// Make available globally
window.buildQueryString = buildQueryString;

