/**
 * Admin Analytics Page - API-based
 */

document.addEventListener('DOMContentLoaded', function() {
    console.log('Analytics page loaded');
    console.log('ADMIN_API_BASE:', typeof ADMIN_API_BASE !== 'undefined' ? ADMIN_API_BASE : 'NOT DEFINED');
    
    // Small delay to ensure admin-api.js is fully loaded
    setTimeout(function() {
        loadAnalytics();
    }, 100);
});

function loadAnalytics() {
    // Function body

    if (typeof ADMIN_API_BASE === 'undefined') {
        console.error('ADMIN_API_BASE is not defined!');
        showError('API configuration error. Please refresh the page.');
        return;
    }
    
    const apiUrl = ADMIN_API_BASE + 'analyticsapi.php';
    console.log('Loading analytics from:', apiUrl);
    showLoading('analyticsContainer');
    
    fetch(apiUrl, {
        method: 'GET',
        credentials: 'same-origin',
        headers: {
            'Accept': 'application/json'
        }
    })
    .then(response => {
        console.log('Analytics API Response status:', response.status);
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            return response.text().then(text => {
                console.error('Non-JSON response:', text.substring(0, 500));
                throw new Error('Server returned non-JSON response');
            });
        }
        if (!response.ok) {
            return response.json().then(data => {
                throw new Error(data.message || `HTTP error! status: ${response.status}`);
            });
        }
        return response.json();
    })
    .then(data => {
        console.log('Analytics API Response:', data);
        hideLoading('analyticsContainer');
        const statsEl = document.getElementById('analyticsStats');
        const revenueEl = document.getElementById('revenueSection');
        const categoryEl = document.getElementById('categorySection');
        if (statsEl) statsEl.style.display = 'grid';
        if (revenueEl) revenueEl.style.display = 'block';
        if (categoryEl) categoryEl.style.display = 'block';
        if (data.success && data.data) {
            populateAnalytics(data.data);
        } else {
            console.error('API returned error:', data);
            showError(data.message || 'Failed to load analytics');
        }
    })
    .catch(error => {
        hideLoading('analyticsContainer');
        console.error('Analytics API Error:', error);
        showError('Error loading analytics: ' + error.message);
    });
}


function populateAnalytics(data) {
    const analytics = data.analytics || {};
    
    // Update main stats
    updateStatCard('total_events', analytics.total_events || 0);
    updateStatCard('total_bookings', analytics.total_bookings || 0);
    updateStatCard('total_revenue', analytics.total_revenue || 0, true);
    updateStatCard('unique_customers', analytics.unique_customers || 0);
    updateStatCard('avg_booking_value', analytics.avg_booking_value || 0, true);
    
    // Populate revenue by month
    populateRevenueChart(data.revenue_by_month || []);
    
    // Populate events by category
    populateCategoryTable(data.events_by_category || []);
}

function populateRevenueChart(revenueData) {
    const tbody = document.querySelector('#revenueTable tbody');
    if (!tbody) return;
    
    if (revenueData.length === 0) {
        tbody.innerHTML = '<tr><td colspan="3" style="text-align: center; padding: 2rem;">No revenue data</td></tr>';
        return;
    }
    
    tbody.innerHTML = revenueData.map(item => `
        <tr>
            <td>${formatMonth(item.month)}</td>
            <td>${formatCurrency(item.revenue || 0)}</td>
            <td>${item.bookings || 0}</td>
        </tr>
    `).join('');
}

function populateCategoryTable(categoryData) {
    const tbody = document.querySelector('#categoryTable tbody');
    if (!tbody) return;
    
    if (categoryData.length === 0) {
        tbody.innerHTML = '<tr><td colspan="3" style="text-align: center; padding: 2rem;">No category data</td></tr>';
        return;
    }
    
    tbody.innerHTML = categoryData.map(item => `
        <tr>
            <td>${escapeHtml(item.category || 'N/A')}</td>
            <td>${item.event_count || 0}</td>
            <td>${formatCurrency(item.revenue || 0)}</td>
        </tr>
    `).join('');
}

function formatMonth(monthString) {
    if (!monthString) return 'N/A';
    const [year, month] = monthString.split('-');
    const date = new Date(year, month - 1);
    return date.toLocaleDateString('en-US', { month: 'long', year: 'numeric' });
}

function updateStatCard(statId, value, isCurrency = false) {
    const elements = document.querySelectorAll(`[data-stat="${statId}"]`);
    elements.forEach(el => {
        el.textContent = isCurrency ? formatCurrency(value) : numberFormat(value);
    });
}

