# Roles & Permissions System

Complete role-based access control (RBAC) system for the admin panel.

## Features

- ✅ Create custom roles
- ✅ Assign permissions to roles
- ✅ Assign roles to users
- ✅ Super Admin and Admin have full access by default
- ✅ Permission-based access control for all admin pages
- ✅ System roles (super_admin, admin) cannot be deleted

## Installation

1. **Run the migration script:**
   ```bash
   php run_roles_migration.php
   ```
   
   Or access via browser:
   ```
   http://localhost/events_belive/run_roles_migration.php?key=migrate_roles_2024
   ```

2. **Verify tables are created:**
   - `roles` - Stores role definitions
   - `permissions` - Stores permission definitions
   - `role_permissions` - Links roles to permissions
   - `user_roles` - Links users to roles

## Default Roles

### Super Admin
- Full access to all features including system settings
- Cannot be deleted or modified (except by super admin)

### Admin
- Full access to all features except system settings
- Cannot be deleted or modified (except by super admin)

## Default Permissions

Permissions are organized by modules:

- **Dashboard**: `dashboard.view`
- **Users**: `users.view`, `users.create`, `users.edit`, `users.delete`
- **Organizers**: `organizers.view`, `organizers.create`, `organizers.edit`, `organizers.delete`, `organizers.approve`
- **Venues**: `venues.view`, `venues.create`, `venues.edit`, `venues.delete`, `venues.approve`
- **Events**: `events.view`, `events.create`, `events.edit`, `events.delete`, `events.approve`, `events.reject`
- **Analytics**: `analytics.view`, `analytics.export`
- **Wallet**: `wallet.view`, `wallet.manage`
- **Settlements**: `settlements.view`, `settlements.process`, `settlements.approve`
- **Loyalty**: `loyalty.view`, `loyalty.manage`
- **CMS**: `cms.view`, `cms.edit`
- **Notifications**: `notifications.view`, `notifications.send`
- **Roles**: `roles.view`, `roles.create`, `roles.edit`, `roles.delete`, `roles.permissions`
- **Settings**: `settings.view`, `settings.edit`

## Usage

### Creating a Role

1. Go to **Roles & Permissions** in admin panel
2. Click **Create Role**
3. Enter role name and slug
4. Select permissions
5. Save

### Assigning Roles to Users

1. Go to **User Management**
2. Select a user
3. Assign roles from the roles dropdown

### Using Permissions in Code

#### Check Permission
```php
require_once __DIR__ . '/includes/permissions.php';

if (hasPermission('events.approve')) {
    // User can approve events
}
```

#### Require Permission
```php
requirePermission('events.delete', 'You cannot delete events');
// Code here will only execute if user has permission
```

#### Check Multiple Permissions
```php
// Check if user has ANY of these permissions
if (hasAnyPermission(['events.edit', 'events.create'])) {
    // User can edit or create events
}

// Check if user has ALL of these permissions
if (hasAllPermissions(['events.view', 'events.edit'])) {
    // User can view and edit events
}
```

#### Check if Super Admin
```php
if (isSuperAdmin()) {
    // Super admin specific code
}
```

### Protecting Pages

```php
require_once __DIR__ . '/../includes/permissions.php';

// Require specific permission
requirePermission('events.view');

// Or check and redirect
if (!hasPermission('events.view')) {
    header('Location: ' . BASE_URL . '/admin/dashboard.php?error=' . urlencode('Access denied'));
    exit;
}
```

### Protecting API Endpoints

```php
require_once __DIR__ . '/../../includes/permissions.php';

requireAdmin();

// Check permission
if (!hasPermission('events.delete') && !isSuperAdmin()) {
    sendError('Insufficient permissions', 403);
}
```

## Database Structure

### Roles Table
- `id` - Primary key
- `name` - Role name
- `slug` - Unique slug (lowercase, numbers, underscores)
- `description` - Role description
- `is_system_role` - Whether it's a system role (cannot be deleted)
- `is_active` - Active status

### Permissions Table
- `id` - Primary key
- `name` - Permission name
- `slug` - Unique permission slug (format: `module.action`)
- `module` - Module name (events, users, etc.)
- `description` - Permission description
- `is_active` - Active status

### Role Permissions Table
- `role_id` - Foreign key to roles
- `permission_id` - Foreign key to permissions
- Unique constraint on (role_id, permission_id)

### User Roles Table
- `user_id` - Foreign key to customers
- `role_id` - Foreign key to roles
- `assigned_by` - Admin who assigned the role
- Unique constraint on (user_id, role_id)

## API Endpoints

### Roles API (`/api/admin/rolesapi.php`)

- `GET ?action=list` - Get all roles
- `GET ?action=get&id=X` - Get single role with permissions
- `GET ?action=permissions` - Get all permissions
- `GET ?action=permissions_by_module` - Get permissions grouped by module
- `POST ?action=create` - Create new role
- `PUT ?action=update` - Update role
- `DELETE ?action=delete&id=X` - Delete role

### User Roles API (`/api/admin/userrolesapi.php`)

- `GET ?action=user_roles&user_id=X` - Get roles for a user
- `GET ?action=users` - Get all users with their roles
- `POST ?action=assign` - Assign role to user
- `DELETE ?action=remove&user_id=X&role_id=Y` - Remove role from user

## Best Practices

1. **Always check permissions** before allowing actions
2. **Use requirePermission()** for critical operations
3. **Super Admin and Admin** have all permissions by default
4. **System roles** should not be modified by non-super-admins
5. **Test permissions** after creating new roles
6. **Document custom permissions** when adding new features

## Troubleshooting

### Permission not working?
1. Check if user has the role assigned
2. Check if role has the permission
3. Check if permission slug is correct
4. Check if user is super_admin or admin (they have all permissions)

### Role not showing?
1. Check if role is active (`is_active = 1`)
2. Check database connection
3. Check API endpoint is accessible

### Migration failed?
1. Check database credentials in `config.php`
2. Check if tables already exist
3. Run migration script again (it's safe to run multiple times)



