# BASE_URL Configuration Guide

## Overview

This project uses **automatic BASE_URL detection** that works seamlessly on both local and production environments without any code changes.

## Single Source of Truth

**`config.php`** is the ONLY file that determines BASE_URL. All other files use the `BASE_URL` constant or helper functions that reference it.

## How It Works

### 1. Auto-Detection Logic (`config.php`)

The system automatically detects the environment:

**PRODUCTION:**
- Domain: `belive-events.videostech.cloud`
- BASE_URL: `https://belive-events.videostech.cloud` (NO `/events_belive`)

**LOCAL:**
- Domain: `localhost`
- BASE_URL: `http://localhost/events_belive` (WITH `/events_belive`)

### 2. Detection Methods (in order of priority)

1. **HTTP_HOST check** - Checks if domain contains `belive-events.videostech.cloud`
2. **REQUEST_URI check** - Extracts `/events_belive` from current URL (local)
3. **SCRIPT_NAME check** - Extracts path from script location
4. **Directory check** - Checks if `events_belive` exists in file path

### 3. Usage in Code

#### PHP Files:
```php
// Always use BASE_URL constant
$url = BASE_URL . '/organizer/dashboard.php';

// Or use helper function (which uses BASE_URL internally)
$url = getCorrectBaseUrl() . '/organizer/dashboard.php';
```

#### JavaScript Files:
```javascript
// BASE_URL is set by PHP in page or auto-detected
const apiUrl = BASE_URL + '/api/organizer/eventsapi.php';
```

## Files Updated

### Core Configuration:
- ✅ `config.php` - Enhanced auto-detection logic
- ✅ `includes/url_helper.php` - Always uses BASE_URL constant
- ✅ `includes/roles.php` - getBaseUrlForRedirect() uses BASE_URL

### Organizer Files:
- ✅ `organizer/dashboard.php` - Removed hardcoded URLs
- ✅ `organizer/pages/events.php` - Removed hardcoded URLs

### JavaScript Files:
- ✅ `admin/assets/js/admin-api.js` - Auto-detects BASE_URL
- ✅ `admin/assets/js/admin-loader.js` - Auto-detects BASE_URL

## Rules

1. **NEVER hardcode** `/events_belive` or `localhost/events_belive` in any file
2. **ALWAYS use** `BASE_URL` constant or `getCorrectBaseUrl()` function
3. **JavaScript** should detect BASE_URL from `window.BASE_URL` (set by PHP) or auto-detect from `window.location`

## Testing

### Local Environment:
- Visit: `http://localhost/events_belive/organizer/dashboard.php`
- BASE_URL should be: `http://localhost/events_belive`

### Production Environment:
- Visit: `https://belive-events.videostech.cloud/organizer/dashboard.php`
- BASE_URL should be: `https://belive-events.videostech.cloud`

## Troubleshooting

If URLs are incorrect:

1. Check `config.php` - BASE_URL should be defined correctly
2. Check browser console - Look for BASE_URL detection logs
3. Verify `$_SERVER['HTTP_HOST']` contains correct domain
4. Check `$_SERVER['REQUEST_URI']` for path detection

## Migration Notes

All hardcoded URLs have been replaced with:
- `BASE_URL` constant (PHP)
- `getCorrectBaseUrl()` function (PHP)
- Auto-detection (JavaScript)

No manual configuration needed - works automatically! 🎉

