# ARTIST Module - Complete Implementation Guide

## Overview

The ARTIST module allows organizers to manage their artists and link them to events. This module follows the same patterns as the EVENTS module and integrates seamlessly with the existing role and permission system.

## Database Structure

### Tables Created

1. **`artists`** - Main artists table
   - `id` - Primary key
   - `organizer_id` - Owner (foreign key to organizers)
   - `name` - Artist name
   - `bio` - Bio/description
   - `profile_image` - Image URL
   - `social_links` - JSON field for social media links
   - `status` - ENUM('published', 'unpublished')
   - `is_active` - Boolean flag
   - `created_at`, `updated_at`, `deleted_at` - Timestamps (soft delete)

2. **`event_artists`** - Pivot table for many-to-many relationship
   - `id` - Primary key
   - `event_id` - Foreign key to events
   - `artist_id` - Foreign key to artists
   - `created_at` - Timestamp
   - Unique constraint on (event_id, artist_id)

## Permissions Added

All permissions are prefixed with `organizer.artists.*`:

- `organizer.artists.view` - View own artists
- `organizer.artists.create` - Create new artists
- `organizer.artists.edit` - Edit own artists
- `organizer.artists.delete` - Delete own artists (soft delete)
- `organizer.artists.publish` - Publish/unpublish own artists

These permissions are automatically assigned to the `organizer_manager` role.

## Files Created

### API Endpoints

1. **`api/organizer/artistsapi.php`**
   - GET - List artists (with filters: status, search)
   - GET ?id=X - Get single artist
   - POST - Create artist
   - POST ?action=update&id=X - Update artist
   - DELETE / POST ?action=delete&id=X - Soft delete artist

2. **`api/organizer/eventartistsapi.php`**
   - GET ?event_id=X - Get artists for an event
   - GET - Get all available artists (for dropdown)
   - POST ?action=link - Link artist(s) to event
   - POST ?action=unlink - Unlink artist from event

### Organizer Pages

1. **`organizer/pages/artists.php`** - List all artists
   - Status filter (all/published/unpublished)
   - Search by name
   - Actions: View, Edit, Delete

2. **`organizer/pages/create-artist.php`** - Create new artist
   - Form fields: name, bio, profile_image, social_links, status
   - Default status: unpublished

3. **`organizer/pages/edit-artist.php`** - Edit existing artist
   - Pre-filled form with artist data
   - Ownership verification

4. **`organizer/pages/artist-detail.php`** - View artist details
   - Full artist information display
   - Social links display
   - Edit button (if permission)

### JavaScript

1. **`organizer/assets/js/organizer-artists.js`**
   - Load artists list
   - Filter and search functionality
   - Delete artist handler
   - Navigation functions

### Integration Files Modified

1. **`organizer/pages/create-event.php`** - Added artist multi-select dropdown
2. **`organizer/pages/edit-event.php`** - Added artist multi-select with current selections
3. **`organizer/pages/event-detail.php`** - Display linked artists
4. **`organizer/includes/sidebar.php`** - Added "My Artists" menu item

## Security Features

1. **Ownership Validation**
   - All API endpoints verify that artists belong to the logged-in organizer
   - Event-artist linking verifies both event and artist ownership
   - Prevents cross-organizer access

2. **Permission Checks**
   - All endpoints check appropriate permissions
   - Pages redirect if user lacks permission

3. **Soft Delete**
   - Artists are soft-deleted (deleted_at timestamp)
   - Deleted artists are excluded from queries
   - Backward compatible if deleted_at column doesn't exist

4. **Input Validation**
   - Required fields validated
   - JSON fields properly encoded/decoded
   - SQL injection prevention via prepared statements

## Usage Examples

### Creating an Artist

1. Navigate to "My Artists" in sidebar
2. Click "Create Artist"
3. Fill in form:
   - Name (required)
   - Bio (optional)
   - Profile Image URL (optional)
   - Social Links (optional)
   - Status (defaults to unpublished)
4. Submit form

### Linking Artists to Events

1. While creating/editing an event
2. Scroll to "Artists" field
3. Select multiple artists (Ctrl/Cmd + Click)
4. Save event
5. Artists are automatically linked

### Viewing Event Artists

1. Open event detail page
2. Scroll to "Artists" section
3. See all linked artists with profile images
4. Click "View Artist" to see full details

## Migration

To set up the ARTIST module:

1. Run the migration:
   ```bash
   mysql -u username -p database_name < migrations/add_artists_module.sql
   ```

2. Or import via phpMyAdmin:
   - Open `migrations/add_artists_module.sql`
   - Execute in phpMyAdmin

The migration will:
- Create `artists` table
- Create `event_artists` pivot table
- Add artist permissions
- Assign permissions to `organizer_manager` role

## API Usage Examples

### List Artists
```javascript
GET /api/organizer/artistsapi.php?status=published&search=john
```

### Create Artist
```javascript
POST /api/organizer/artistsapi.php
{
  "name": "John Doe",
  "bio": "Singer and songwriter",
  "profile_image": "https://example.com/image.jpg",
  "social_links": {
    "website": "https://johndoe.com",
    "facebook": "https://facebook.com/johndoe"
  }
}
```

### Link Artists to Event
```javascript
POST /api/organizer/eventartistsapi.php?action=link
{
  "event_id": 123,
  "artist_ids": [1, 2, 3]
}
```

## Notes

- Artists are organizer-specific (cannot see other organizers' artists)
- Artists can be linked to multiple events
- Events can have multiple artists
- Soft delete preserves data integrity
- All operations respect role-based permissions
- Backward compatible with existing codebase

## Future Enhancements

Potential additions:
- Artist image upload (currently URL only)
- Artist categories/tags
- Artist performance history
- Artist availability calendar
- Bulk artist operations

