<?php
/**
 * Test script to verify prerequisites before running parallel_findall_to_db.php
 */

echo "=== Testing Prerequisites ===\n\n";

$errors = [];
$warnings = [];

// 1. Check PHP version
echo "1. PHP Version: " . PHP_VERSION . "\n";
if (version_compare(PHP_VERSION, '7.4.0', '<')) {
    $errors[] = "PHP 7.4+ required";
}

// 2. Check required extensions
echo "2. Checking PHP extensions:\n";
$required = ['pdo', 'pdo_mysql', 'curl', 'json'];
foreach ($required as $ext) {
    $loaded = extension_loaded($ext);
    echo "   - $ext: " . ($loaded ? "✓" : "✗ MISSING") . "\n";
    if (!$loaded) {
        $errors[] = "Missing extension: $ext";
    }
}

// 3. Check environment variable
echo "\n3. Checking PARALLEL_API_KEY:\n";
$apiKey = getenv('PARALLEL_API_KEY');
if (empty($apiKey)) {
    $errors[] = "PARALLEL_API_KEY environment variable not set";
    echo "   ✗ Not set\n";
} else {
    echo "   ✓ Set (length: " . strlen($apiKey) . ")\n";
}

// 4. Test database connection
echo "\n4. Testing database connection:\n";
$dbHost = 'localhost';
$dbName = 'db_belive_events';
$dbUser = 'root';
$dbPass = 'mtv';

try {
    $pdo = new PDO(
        "mysql:host={$dbHost};dbname={$dbName};charset=utf8mb4",
        $dbUser,
        $dbPass,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    echo "   ✓ Connected successfully\n";
    
    // Check if events table exists
    echo "\n5. Checking events table:\n";
    $stmt = $pdo->query("SHOW TABLES LIKE 'events'");
    if ($stmt->rowCount() === 0) {
        $errors[] = "Table 'events' does not exist";
        echo "   ✗ Table 'events' not found\n";
    } else {
        echo "   ✓ Table 'events' exists\n";
        
        // Check table structure
        echo "\n6. Checking table structure:\n";
        $stmt = $pdo->query("DESCRIBE events");
        $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
        $requiredColumns = [
            'id', 'name', 'start_date', 'end_date', 'attendees_range', 
            'venue', 'city', 'country', 'ticket_available', 'pricing', 
            'description', 'attendee_profiles', 'sponsors', 'event_type', 
            'website_url', 'raw_candidate_json', 'created_at', 'updated_at'
        ];
        
        foreach ($requiredColumns as $col) {
            $exists = in_array($col, $columns, true);
            echo "   - $col: " . ($exists ? "✓" : "✗ MISSING") . "\n";
            if (!$exists) {
                $errors[] = "Missing column: $col";
            }
        }
        
        // Check for PRIMARY KEY on id
        echo "\n7. Checking PRIMARY KEY constraint:\n";
        $stmt = $pdo->query("SHOW KEYS FROM events WHERE Key_name = 'PRIMARY'");
        $pk = $stmt->fetch();
        if (!$pk || $pk['Column_name'] !== 'id') {
            $warnings[] = "PRIMARY KEY on 'id' column is required for ON DUPLICATE KEY UPDATE to work";
            echo "   ⚠ PRIMARY KEY on 'id' not found or incorrect\n";
        } else {
            echo "   ✓ PRIMARY KEY on 'id' exists\n";
        }
    }
    
} catch (PDOException $e) {
    $errors[] = "Database connection failed: " . $e->getMessage();
    echo "   ✗ Connection failed: " . $e->getMessage() . "\n";
}

// Summary
echo "\n=== Summary ===\n";
if (empty($errors) && empty($warnings)) {
    echo "✓ All checks passed! The script should work.\n";
    exit(0);
}

if (!empty($warnings)) {
    echo "\n⚠ Warnings:\n";
    foreach ($warnings as $w) {
        echo "  - $w\n";
    }
}

if (!empty($errors)) {
    echo "\n✗ Errors (must fix before running):\n";
    foreach ($errors as $e) {
        echo "  - $e\n";
    }
    exit(1);
}

exit(0);

