// Global state
let currentFilter = 'all';
let allEvents = [];

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    initializeFilters();
    
    // Check URL parameters for filters
    const urlParams = new URLSearchParams(window.location.search);
    const typeParam = urlParams.get('type');
    const countryParam = urlParams.get('country');
    
    if (typeParam) {
        currentFilter = typeParam;
        updateActiveFilter();
        loadEvents(typeParam, countryParam);
    } else if (countryParam) {
        // Filter by country
        loadEvents(null, countryParam);
    } else {
        // Load all events
        loadEvents();
    }
});

// Initialize filter buttons
function initializeFilters() {
    const filterTabs = document.querySelectorAll('.filter-tab');
    filterTabs.forEach(tab => {
        tab.addEventListener('click', function() {
            currentFilter = this.getAttribute('data-filter');
            updateActiveFilter();
            filterEvents();
        });
    });
}

// Update active filter button
function updateActiveFilter() {
    const filterTabs = document.querySelectorAll('.filter-tab');
    filterTabs.forEach(tab => {
        if (tab.getAttribute('data-filter') === currentFilter) {
            tab.classList.add('active');
        } else {
            tab.classList.remove('active');
        }
    });
}

// Load events from API
function loadEvents(eventType = null, country = null) {
    const container = document.getElementById('eventsContainer');
    const loadingIndicator = document.getElementById('loadingIndicator');
    const noEvents = document.getElementById('noEvents');
    
    // Show loading
    if (container) container.innerHTML = '';
    if (loadingIndicator) loadingIndicator.style.display = 'block';
    if (noEvents) noEvents.style.display = 'none';
    
    // Build query string
    let queryParams = [];
    if (eventType && eventType !== 'all') {
        queryParams.push('type=' + encodeURIComponent(eventType));
    }
    if (country) {
        queryParams.push('country=' + encodeURIComponent(country));
    }
    
    const queryString = queryParams.length > 0 ? '?' + queryParams.join('&') : '';
    
    // Make AJAX request
    fetch('api/events.php' + queryString)
        .then(response => response.json())
        .then(data => {
            if (loadingIndicator) loadingIndicator.style.display = 'none';
            
            if (data.success && data.events && data.events.length > 0) {
                allEvents = data.events;
                displayEvents(data.events);
                updateEventCount(data.events.length);
            } else {
                if (noEvents) noEvents.style.display = 'block';
                updateEventCount(0);
            }
        })
        .catch(error => {
            console.error('Error loading events:', error);
            if (loadingIndicator) loadingIndicator.style.display = 'none';
            if (noEvents) {
                noEvents.style.display = 'block';
                noEvents.innerHTML = '<p>Error loading events. Please try again later.</p>';
            }
            updateEventCount(0);
        });
}

// Filter events based on current filter - reload from API
function filterEvents() {
    // Reload events from API with the current filter
    loadEvents(currentFilter === 'all' ? null : currentFilter, null);
}

// Display events in the grid
function displayEvents(events) {
    const container = document.getElementById('eventsContainer');
    if (!container) return;
    
    if (events.length === 0) {
        container.innerHTML = '<div class="no-events"><p>No events found. Try adjusting your filters.</p></div>';
        return;
    }
    
    container.innerHTML = events.map(event => createEventCard(event)).join('');
}

// Create event card HTML with improved design
function createEventCard(event) {
    const startDate = event.start_date ? new Date(event.start_date) : null;
    const endDate = event.end_date ? new Date(event.end_date) : null;
    
    // Format date header (e.g., "OCT 14")
    const dateHeader = startDate ? formatDateHeader(startDate) : '';
    
    // Format date range (e.g., "Oct 14, 2024 – Oct 18, 2024")
    const dateRange = formatDateRange(startDate, endDate);
    
    // Event type - capitalize first letter
    const eventType = capitalizeFirst(event.event_type || 'Event');
    
    // Location
    const country = event.country || '';
    const city = event.city || '';
    const location = country ? (city ? `${city}, ${country}` : country) : 'TBA';
    
    // Venue
    const venue = event.venue && event.venue !== 'TBA' ? event.venue : '';
    const fullLocation = venue ? `${venue}, ${location}` : location;
    
    // Attendees
    const attendees = event.attendees_range || '1000+';
    const attendeesFormatted = formatAttendees(attendees);
    
    // Ticket availability
    const hasTickets = event.ticket_available;
    const pricing = event.pricing || '';
    
    // Description - truncate if too long
    const description = event.description ? truncateText(event.description, 150) : '';
    
    return `
        <div class="event-card" onclick="window.location.href='event-detail.html?id=${encodeURIComponent(event.id)}'">
            ${dateHeader ? `<div class="event-date-header">${dateHeader}</div>` : ''}
            <div class="event-meta">
                <span class="event-type-badge">${escapeHtml(eventType)}</span>
                ${country ? `<span class="event-location-badge">${escapeHtml(country)}</span>` : ''}
            </div>
            <h3 class="event-name">${escapeHtml(event.name || 'Untitled Event')}</h3>
            ${description ? `<p class="event-description">${escapeHtml(description)}</p>` : ''}
            <div class="event-info">
                ${dateRange ? `
                    <div class="event-info-item">
                        <span class="info-icon">📅</span>
                        <span>${dateRange}</span>
                    </div>
                ` : ''}
                ${fullLocation ? `
                    <div class="event-info-item">
                        <span class="info-icon">📍</span>
                        <span>${escapeHtml(fullLocation)}</span>
                    </div>
                ` : ''}
                ${attendeesFormatted ? `
                    <div class="event-info-item">
                        <span class="info-icon">👥</span>
                        <span>${escapeHtml(attendeesFormatted)}</span>
                    </div>
                ` : ''}
            </div>
            ${hasTickets ? `
                <div class="event-ticket-badge">
                    <span class="ticket-icon">🎟</span>
                    <span>TICKETS AVAILABLE</span>
                    ${pricing ? `<span class="ticket-price">From ${escapeHtml(pricing)}</span>` : ''}
                </div>
            ` : ''}
            ${event.website_url ? `
                <a href="${escapeHtml(event.website_url)}" target="_blank" rel="noopener noreferrer" class="event-link" onclick="event.stopPropagation()">
                    Visit Website →
                </a>
            ` : ''}
        </div>
    `;
}

// Format date header (e.g., "OCT 14")
function formatDateHeader(date) {
    const months = ['JAN', 'FEB', 'MAR', 'APR', 'MAY', 'JUN', 'JUL', 'AUG', 'SEP', 'OCT', 'NOV', 'DEC'];
    return `${months[date.getMonth()]} ${date.getDate()}`;
}

// Format date range
function formatDateRange(startDate, endDate) {
    if (!startDate) return '';
    
    const formatDate = (date) => {
        return date.toLocaleDateString('en-US', {
            month: 'short',
            day: 'numeric',
            year: 'numeric'
        });
    };
    
    const start = formatDate(startDate);
    if (endDate && endDate.getTime() !== startDate.getTime()) {
        const end = formatDate(endDate);
        return `${start} – ${end}`;
    }
    return start;
}

// Format attendees
function formatAttendees(attendees) {
    if (!attendees) return '';
    // If it's already formatted nicely, return as is
    if (attendees.includes('+') || attendees.includes('-') || attendees.includes(',')) {
        return attendees + ' attendees';
    }
    return attendees + ' attendees';
}

// Truncate text
function truncateText(text, maxLength) {
    if (!text || text.length <= maxLength) return text;
    return text.substring(0, maxLength).trim() + '...';
}

// Capitalize first letter
function capitalizeFirst(str) {
    if (!str) return '';
    return str.charAt(0).toUpperCase() + str.slice(1).toLowerCase();
}

// Format date
function formatDate(dateString) {
    if (!dateString) return 'TBA';
    
    try {
        const date = new Date(dateString);
        if (isNaN(date.getTime())) return dateString;
        
        return date.toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });
    } catch (e) {
        return dateString;
    }
}

// Update event count
function updateEventCount(count) {
    const countElement = document.getElementById('filtersCount');
    if (countElement) {
        countElement.textContent = `${count} event${count !== 1 ? 's' : ''} found`;
    }
}

// Escape HTML to prevent XSS
function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
