<?php
/**
 * User Activity Helper Functions
 * Fetch and store user activity from social media platforms
 */

require_once __DIR__ . '/auth.php';

/**
 * Fetch and store Google user activity
 */
function fetchGoogleActivity($accessToken, $customerId) {
    try {
        $pdo = getDB();
        $activityData = [];
        
        // Fetch user profile with additional info
        $profileUrl = 'https://www.googleapis.com/oauth2/v2/userinfo?access_token=' . urlencode($accessToken);
        $ch = curl_init($profileUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200) {
            $profileData = json_decode($response, true);
            $activityData['profile'] = $profileData;
            
            // Extract interests from profile
            $interests = [];
            if (!empty($profileData['locale'])) {
                $interests[] = $profileData['locale'];
            }
            
            // Store activity - ensure interests are strings
            $interestsClean = array_filter(array_map('trim', $interests), function($item) {
                return is_string($item) && !empty($item);
            });
            storeUserActivity($customerId, 'profile', $activityData, $interestsClean, 'google');
        }
        
        // Try to fetch YouTube activity (if available)
        try {
            $youtubeUrl = 'https://www.googleapis.com/youtube/v3/channels?part=snippet&mine=true&access_token=' . urlencode($accessToken);
            $ch = curl_init($youtubeUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200) {
                $youtubeData = json_decode($response, true);
                if (!empty($youtubeData['items'])) {
                    $activityData['youtube'] = $youtubeData;
                    storeUserActivity($customerId, 'youtube', $youtubeData, [], 'google');
                }
            }
        } catch (Exception $e) {
            error_log('YouTube API error: ' . $e->getMessage());
        }
        
        return $activityData;
    } catch (Exception $e) {
        error_log('Error fetching Google activity: ' . $e->getMessage());
        return [];
    }
}

/**
 * Fetch and store Facebook user activity
 */
function fetchFacebookActivity($accessToken, $customerId) {
    try {
        $pdo = getDB();
        $activityData = [];
        $interests = [];
        
        // Fetch user profile with extended permissions
        $fields = 'id,name,email,first_name,last_name,picture,location,hometown,education,work,likes{name,category},interests{name}';
        $profileUrl = 'https://graph.facebook.com/v18.0/me?fields=' . urlencode($fields) . '&access_token=' . urlencode($accessToken);
        
        $ch = curl_init($profileUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200) {
            $profileData = json_decode($response, true);
            $activityData['profile'] = $profileData;
            
            // Extract interests from likes
            if (!empty($profileData['likes']['data'])) {
                foreach ($profileData['likes']['data'] as $like) {
                    if (!empty($like['name'])) {
                        $interests[] = $like['name'];
                    }
                    if (!empty($like['category'])) {
                        $interests[] = $like['category'];
                    }
                }
            }
            
            // Extract interests
            if (!empty($profileData['interests']['data'])) {
                foreach ($profileData['interests']['data'] as $interest) {
                    if (!empty($interest['name'])) {
                        $interests[] = $interest['name'];
                    }
                }
            }
            
            // Extract location
            if (!empty($profileData['location']['name'])) {
                $interests[] = $profileData['location']['name'];
            }
            
            // Extract education/work info
            if (!empty($profileData['education'])) {
                foreach ($profileData['education'] as $edu) {
                    if (!empty($edu['school']['name'])) {
                        $interests[] = $edu['school']['name'];
                    }
                    if (!empty($edu['type'])) {
                        $interests[] = $edu['type'];
                    }
                }
            }
            
            if (!empty($profileData['work'])) {
                foreach ($profileData['work'] as $work) {
                    if (!empty($work['employer']['name'])) {
                        $interests[] = $work['employer']['name'];
                    }
                    if (!empty($work['position']['name'])) {
                        $interests[] = $work['position']['name'];
                    }
                }
            }
            
            // Store activity - ensure interests are strings
            $interestsClean = [];
            foreach ($interests as $interest) {
                if (is_string($interest) && !empty(trim($interest))) {
                    $interestsClean[] = trim($interest);
                } elseif (is_array($interest)) {
                    foreach ($interest as $item) {
                        if (is_string($item) && !empty(trim($item))) {
                            $interestsClean[] = trim($item);
                        }
                    }
                }
            }
            storeUserActivity($customerId, 'profile', $activityData, array_unique($interestsClean), 'facebook');
        }
        
        return $activityData;
    } catch (Exception $e) {
        error_log('Error fetching Facebook activity: ' . $e->getMessage());
        return [];
    }
}

/**
 * Fetch and store LinkedIn user activity
 */
function fetchLinkedInActivity($accessToken, $customerId) {
    try {
        $pdo = getDB();
        $activityData = [];
        $interests = [];
        
        // Fetch user profile with extended info
        $profileUrl = 'https://api.linkedin.com/v2/userinfo';
        $ch = curl_init($profileUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Authorization: Bearer ' . $accessToken]);
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200) {
            $profileData = json_decode($response, true);
            $activityData['profile'] = $profileData;
            
            // Extract basic info
            if (!empty($profileData['given_name'])) {
                $interests[] = $profileData['given_name'];
            }
            if (!empty($profileData['family_name'])) {
                $interests[] = $profileData['family_name'];
            }
            if (!empty($profileData['locale'])) {
                $interests[] = $profileData['locale'];
            }
        }
        
        // Try to fetch profile with more details
        try {
            $detailedUrl = 'https://api.linkedin.com/v2/me?projection=(id,firstName,lastName,profilePicture(displayImage~:playableStreams))';
            $ch = curl_init($detailedUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Authorization: Bearer ' . $accessToken]);
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200) {
                $detailedData = json_decode($response, true);
                $activityData['detailed'] = $detailedData;
            }
        } catch (Exception $e) {
            error_log('LinkedIn detailed profile error: ' . $e->getMessage());
        }
        
        // Store activity - ensure interests are strings
        $interestsClean = [];
        foreach ($interests as $interest) {
            if (is_string($interest) && !empty(trim($interest))) {
                $interestsClean[] = trim($interest);
            } elseif (is_array($interest)) {
                // Flatten nested arrays
                foreach ($interest as $item) {
                    if (is_string($item) && !empty(trim($item))) {
                        $interestsClean[] = trim($item);
                    }
                }
            }
        }
        
        if (!empty($activityData)) {
            storeUserActivity($customerId, 'profile', $activityData, array_unique($interestsClean), 'linkedin');
        }
        
        return $activityData;
    } catch (Exception $e) {
        error_log('Error fetching LinkedIn activity: ' . $e->getMessage());
        return [];
    }
}

/**
 * Store user activity in database
 */
function storeUserActivity($customerId, $activityType, $activityData, $keywords = [], $source) {
    try {
        $pdo = getDB();
        
        // Check if user_activity table exists
        try {
            $pdo->query("SELECT 1 FROM user_activity LIMIT 1");
        } catch (Exception $e) {
            // Table doesn't exist yet, skip storing
            error_log('user_activity table does not exist. Please run database migration.');
            return;
        }
        
        // Convert keywords array to string - ensure all values are strings
        $keywordsClean = [];
        foreach ($keywords as $keyword) {
            if (is_string($keyword) && !empty(trim($keyword))) {
                $keywordsClean[] = trim($keyword);
            } elseif (is_array($keyword)) {
                // Flatten nested arrays
                foreach ($keyword as $item) {
                    if (is_string($item) && !empty(trim($item))) {
                        $keywordsClean[] = trim($item);
                    }
                }
            }
        }
        $keywordsStr = !empty($keywordsClean) ? implode(', ', array_unique($keywordsClean)) : null;
        
        // Check if activity exists
        $checkStmt = $pdo->prepare("
            SELECT id FROM user_activity 
            WHERE customer_id = :customer_id AND activity_type = :activity_type AND source = :source
        ");
        $checkStmt->execute([
            ':customer_id' => $customerId,
            ':activity_type' => $activityType,
            ':source' => $source
        ]);
        $existing = $checkStmt->fetch();
        
        if ($existing) {
            // Update existing
            $stmt = $pdo->prepare("
                UPDATE user_activity 
                SET activity_data = :activity_data,
                    keywords = :keywords,
                    updated_at = NOW()
                WHERE id = :id
            ");
            $stmt->execute([
                ':id' => $existing['id'],
                ':activity_data' => json_encode($activityData),
                ':keywords' => $keywordsStr
            ]);
        } else {
            // Insert new
            $stmt = $pdo->prepare("
                INSERT INTO user_activity (customer_id, activity_type, activity_data, keywords, source)
                VALUES (:customer_id, :activity_type, :activity_data, :keywords, :source)
            ");
            $stmt->execute([
                ':customer_id' => $customerId,
                ':activity_type' => $activityType,
                ':activity_data' => json_encode($activityData),
                ':keywords' => $keywordsStr,
                ':source' => $source
            ]);
        }
        
        // Update customer table with interests
        if (!empty($keywords)) {
            $stmt = $pdo->prepare("
                UPDATE customers 
                SET interests = :interests,
                    activity_data = :activity_data
                WHERE id = :id
            ");
            
            $existingInterests = [];
            $stmtCheck = $pdo->prepare("SELECT interests FROM customers WHERE id = :id");
            $stmtCheck->execute([':id' => $customerId]);
            $existing = $stmtCheck->fetch();
            if ($existing && !empty($existing['interests'])) {
                $existingInterests = array_map('trim', explode(',', $existing['interests']));
            }
            
            // Ensure keywords are strings before merging
            $keywordsClean = [];
            foreach ($keywords as $keyword) {
                if (is_string($keyword) && !empty(trim($keyword))) {
                    $keywordsClean[] = trim($keyword);
                } elseif (is_array($keyword)) {
                    foreach ($keyword as $item) {
                        if (is_string($item) && !empty(trim($item))) {
                            $keywordsClean[] = trim($item);
                        }
                    }
                }
            }
            $allInterests = array_unique(array_merge($existingInterests, $keywordsClean));
            $interestsStr = implode(', ', $allInterests);
            
            $stmt->execute([
                ':id' => $customerId,
                ':interests' => $interestsStr,
                ':activity_data' => json_encode($activityData)
            ]);
        }
        
        // Analyze and store preferences
        analyzeAndStorePreferences($customerId, $activityData, $keywords, $source);
        
    } catch (Exception $e) {
        error_log('Error storing user activity: ' . $e->getMessage());
    }
}

/**
 * Analyze activity data and store preferences
 */
function analyzeAndStorePreferences($customerId, $activityData, $keywords, $source) {
    try {
        $pdo = getDB();
        
        // Check if user_preferences table exists
        try {
            $pdo->query("SELECT 1 FROM user_preferences LIMIT 1");
        } catch (Exception $e) {
            // Table doesn't exist yet, skip storing
            error_log('user_preferences table does not exist. Please run database migration.');
            return;
        }
        
        $preferences = [];
        
        // Extract event type preferences from keywords
        $eventTypeKeywords = [
            'conference' => ['conference', 'meetup', 'summit', 'forum', 'symposium'],
            'exhibition' => ['exhibition', 'expo', 'trade show', 'fair', 'showcase'],
            'festival' => ['festival', 'celebration', 'carnival', 'gathering'],
            'workshop' => ['workshop', 'training', 'seminar', 'course', 'tutorial'],
            'sports' => ['sports', 'football', 'basketball', 'tennis', 'cricket', 'soccer'],
            'concert' => ['concert', 'music', 'band', 'performance', 'live'],
            'networking' => ['networking', 'meetup', 'community', 'social']
        ];
        
        // Ensure keywords are strings before processing
        $keywordsClean = [];
        foreach ($keywords as $keyword) {
            if (is_string($keyword) && !empty(trim($keyword))) {
                $keywordsClean[] = trim($keyword);
            } elseif (is_array($keyword)) {
                foreach ($keyword as $item) {
                    if (is_string($item) && !empty(trim($item))) {
                        $keywordsClean[] = trim($item);
                    }
                }
            }
        }
        
        foreach ($keywordsClean as $keyword) {
            $keywordLower = strtolower($keyword);
            foreach ($eventTypeKeywords as $eventType => $typeKeywords) {
                foreach ($typeKeywords as $typeKeyword) {
                    if (strpos($keywordLower, $typeKeyword) !== false) {
                        $preferences[] = [
                            'type' => 'event_type',
                            'value' => $eventType,
                            'weight' => 1.0,
                            'source' => $source
                        ];
                        break 2;
                    }
                }
            }
        }
        
        // Extract location preferences
        if (!empty($activityData['profile']['location'])) {
            $location = $activityData['profile']['location'];
            if (is_string($location)) {
                $preferences[] = [
                    'type' => 'location',
                    'value' => $location,
                    'weight' => 1.5,
                    'source' => $source
                ];
            } elseif (is_array($location) && !empty($location['name'])) {
                $preferences[] = [
                    'type' => 'location',
                    'value' => $location['name'],
                    'weight' => 1.5,
                    'source' => $source
                ];
            }
        }
        
        // Store preferences
        foreach ($preferences as $pref) {
            $stmt = $pdo->prepare("
                INSERT INTO user_preferences (customer_id, preference_type, preference_value, weight, source)
                VALUES (:customer_id, :preference_type, :preference_value, :weight, :source)
                ON DUPLICATE KEY UPDATE
                    weight = GREATEST(weight, VALUES(weight)),
                    updated_at = NOW()
            ");
            
            $stmt->execute([
                ':customer_id' => $customerId,
                ':preference_type' => $pref['type'],
                ':preference_value' => $pref['value'],
                ':weight' => $pref['weight'],
                ':source' => $pref['source']
            ]);
        }
        
    } catch (Exception $e) {
        error_log('Error analyzing preferences: ' . $e->getMessage());
    }
}

/**
 * Get user preferences for event recommendations
 */
function getUserPreferences($customerId) {
    try {
        $pdo = getDB();
        
        $stmt = $pdo->prepare("
            SELECT preference_type, preference_value, weight, source
            FROM user_preferences
            WHERE customer_id = :customer_id
            ORDER BY weight DESC
        ");
        
        $stmt->execute([':customer_id' => $customerId]);
        return $stmt->fetchAll();
        
    } catch (Exception $e) {
        error_log('Error getting user preferences: ' . $e->getMessage());
        return [];
    }
}
