// Get event ID from URL
const urlParams = new URLSearchParams(window.location.search);
const eventId = urlParams.get('id');

let eventData = null;

// Load event details on page load
document.addEventListener('DOMContentLoaded', function() {
    if (!eventId) {
        document.body.innerHTML = '<div class="container"><h1>Event not found</h1><p><a href="events.html">Back to Events</a></p></div>';
        return;
    }
    
    // Check for payment status in URL
    const paymentStatus = urlParams.get('payment');
    const sessionId = urlParams.get('session_id');
    
    if (paymentStatus === 'success' && sessionId) {
        verifyPayment(sessionId);
    } else if (paymentStatus === 'success') {
        showPaymentSuccessMessage();
    } else if (paymentStatus === 'cancelled') {
        showPaymentCancelledMessage();
    }
    
    loadEventDetails(eventId);
});

// Load event details from API
function loadEventDetails(id) {
    fetch(`api/event.php?id=${encodeURIComponent(id)}`)
        .then(response => response.json())
        .then(data => {
            if (data.success && data.event) {
                eventData = data.event;
                populateEventDetails(data.event);
            } else {
                document.body.innerHTML = '<div class="container"><h1>Event not found</h1><p><a href="events.html">Back to Events</a></p></div>';
            }
        })
        .catch(error => {
            console.error('Error loading event:', error);
            document.body.innerHTML = '<div class="container"><h1>Error loading event</h1><p><a href="events.html">Back to Events</a></p></div>';
        });
}

// Populate event details on the page
function populateEventDetails(event) {
    // Event type badge
    const eventType = capitalizeFirst(event.event_type || 'Event');
    document.getElementById('detailEventType').textContent = eventType.toUpperCase();
    
    // Ticket badge
    if (event.ticket_available) {
        document.getElementById('detailTicketBadge').style.display = 'inline-flex';
    }
    
    // Title
    document.getElementById('eventDetailTitle').textContent = event.name || 'Untitled Event';
    
    // Description
    document.getElementById('eventDetailDescription').textContent = event.description || 'No description available.';
    
    // Date and location
    const startDate = event.start_date ? new Date(event.start_date) : null;
    const endDate = event.end_date ? new Date(event.end_date) : null;
    const dateRange = formatDetailDateRange(startDate, endDate);
    document.getElementById('eventDetailDate').textContent = dateRange;
    
    const city = event.city || '';
    const country = event.country || '';
    const venue = event.venue && event.venue !== 'TBA' ? event.venue : '';
    const location = venue ? `${venue}, ${city ? city + ', ' : ''}${country}` : (city ? `${city}, ${country}` : country || 'TBA');
    document.getElementById('eventDetailLocation').textContent = location;
    
    // Statistics cards
    populateStatsCards(event, startDate, endDate);
    
    // Who Attends
    document.getElementById('whoAttends').textContent = event.attendee_profiles || 'General public and event enthusiasts';
    
    // Sponsors
    populateSponsors(event.sponsors || '');
    
    // Sidebar
    document.getElementById('sidebarDate').textContent = formatSidebarDate(startDate);
    const price = event.pricing || (event.ticket_available ? 'Free' : 'TBA');
    document.getElementById('sidebarPrice').textContent = price;
    
    const locationLabel = venue || city || 'Location';
    const locationValue = city && country ? `${city}, ${country}` : country || 'TBA';
    document.getElementById('sidebarLocationLabel').textContent = locationLabel;
    document.getElementById('sidebarLocationValue').textContent = locationValue;
}

// Populate statistics cards
function populateStatsCards(event, startDate, endDate) {
    const attendees = event.attendees_range || '1000+';
    const country = event.country || 'TBA';
    
    // Calculate duration
    let duration = 'TBA';
    if (startDate && endDate) {
        const diffTime = Math.abs(endDate - startDate);
        const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24)) + 1;
        duration = diffDays === 1 ? '1 Day' : `${diffDays} Days`;
    } else if (startDate) {
        duration = '1 Day';
    }
    
    const price = event.pricing || (event.ticket_available ? 'Free' : 'TBA');
    
    const stats = [
        { icon: '👥', value: attendees, label: 'Expected Attendees' },
        { icon: '🏢', value: country, label: 'Location' },
        { icon: '⏰', value: duration, label: 'Duration' },
        { icon: '💰', value: price, label: 'Starting Price' }
    ];
    
    const statsGrid = document.getElementById('eventStatsGrid');
    statsGrid.innerHTML = stats.map(stat => `
        <div class="stat-card">
            <div class="stat-icon">${stat.icon}</div>
            <div class="stat-value">${escapeHtml(stat.value)}</div>
            <div class="stat-label">${escapeHtml(stat.label)}</div>
        </div>
    `).join('');
}

// Populate sponsors
function populateSponsors(sponsorsStr) {
    const sponsorsList = document.getElementById('sponsorsList');
    
    if (!sponsorsStr || sponsorsStr.trim() === '') {
        sponsorsList.innerHTML = '<p class="no-sponsors">No sponsors listed</p>';
        return;
    }
    
    const sponsors = sponsorsStr.split(',').map(s => s.trim()).filter(s => s);
    
    if (sponsors.length === 0) {
        sponsorsList.innerHTML = '<p class="no-sponsors">No sponsors listed</p>';
        return;
    }
    
    sponsorsList.innerHTML = sponsors.map(sponsor => `
        <span class="sponsor-tag">${escapeHtml(sponsor)}</span>
    `).join('');
}

// Format date range for detail page
function formatDetailDateRange(startDate, endDate) {
    if (!startDate) return 'Date TBA';
    
    const formatDate = (date) => {
        return date.toLocaleDateString('en-US', {
            weekday: 'long',
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });
    };
    
    const start = formatDate(startDate);
    if (endDate && endDate.getTime() !== startDate.getTime()) {
        const end = formatDate(endDate);
        return `${start} - ${end}`;
    }
    return start;
}

// Format sidebar date
function formatSidebarDate(date) {
    if (!date) return 'TBA';
    return date.toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
}

// Handle Get Tickets button
async function handleGetTickets() {
    if (!eventData) {
        alert('Event information not loaded. Please refresh the page.');
        return;
    }

    const pricing = eventData.pricing ? eventData.pricing.toString().trim() : '';

    // Extract numeric amount from pricing
    const numberMatch = pricing.match(/\d+(\.\d+)?/);
    const amount = numberMatch ? parseFloat(numberMatch[0]) : 0;

    // ❌ No valid amount → redirect to website
    if (!amount || amount <= 0) {
        if (eventData.website_url) {
            window.open(eventData.website_url, '_blank', 'noopener,noreferrer');
        } else {
            alert('Ticket information not available. Please check the event website.');
        }
        return;
    }

    // ✅ Amount exists → proceed with payment
    const btn = document.getElementById('getTicketsBtn');
    const originalText = btn.innerHTML;

    try {
        btn.disabled = true;
        btn.innerHTML = '<span>Processing...</span>';

        const response = await fetch('api/payment/create-checkout.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                event_id: eventData.id,
                quantity: 1,
            }),
        });

        const data = await response.json();

        if (data.success && data.checkout_url) {
            window.location.href = data.checkout_url;
        } else {
            throw new Error(data.error || 'Failed to create payment session');
        }

    } catch (error) {
        console.error('Payment error:', error);
        alert('Unable to process payment. Please try again later.\n\nError: ' + error.message);
        btn.disabled = false;
        btn.innerHTML = originalText;
    }
}


// Verify payment status
async function verifyPayment(sessionId) {
    try {
        const response = await fetch(`api/payment/verify.php?session_id=${encodeURIComponent(sessionId)}`);
        const data = await response.json();
        
        if (data.success && data.payment) {
            if (data.payment.payment_status === 'completed') {
                showPaymentSuccessMessage(data.payment);
            } else {
                showPaymentPendingMessage(data.payment);
            }
        }
    } catch (error) {
        console.error('Payment verification error:', error);
    }
}

// Show payment success message
function showPaymentSuccessMessage(payment) {
    const message = payment 
        ? `Payment successful! You have purchased ${payment.ticket_quantity} ticket(s) for ${payment.event_name}. Amount paid: ${payment.currency} ${payment.amount}.`
        : 'Payment successful! Your tickets have been purchased.';
    
    alert(message);
    
    // Clean URL
    const cleanUrl = window.location.pathname + '?id=' + encodeURIComponent(eventId);
    window.history.replaceState({}, document.title, cleanUrl);
}

// Show payment pending message
function showPaymentPendingMessage(payment) {
    alert('Your payment is being processed. You will receive a confirmation email shortly.');
}

// Show payment cancelled message
function showPaymentCancelledMessage() {
    alert('Payment was cancelled. You can try again when you\'re ready.');
    
    // Clean URL
    const cleanUrl = window.location.pathname + '?id=' + encodeURIComponent(eventId);
    window.history.replaceState({}, document.title, cleanUrl);
}

// Handle Share Event button
function handleShareEvent() {
    if (navigator.share) {
        navigator.share({
            title: eventData.name,
            text: eventData.description,
            url: window.location.href
        }).catch(err => console.log('Error sharing', err));
    } else {
        // Fallback: copy to clipboard
        navigator.clipboard.writeText(window.location.href).then(() => {
            alert('Event link copied to clipboard!');
        }).catch(() => {
            // Fallback: show URL
            prompt('Copy this link:', window.location.href);
        });
    }
}

// Capitalize first letter
function capitalizeFirst(str) {
    if (!str) return '';
    return str.charAt(0).toUpperCase() + str.slice(1).toLowerCase();
}

// Escape HTML
function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
