<?php
/**
 * User Preferences API Endpoint
 * Returns user's social media activity and preferences
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: Content-Type');

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/user_activity.php';

// Check if user is logged in
if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'error' => 'Unauthorized'
    ]);
    exit;
}

try {
    $pdo = getDB();
    $customerId = $_SESSION['user_id'];
    
    // Get user preferences
    $preferences = getUserPreferences($customerId);
    
    // Get user activity
    $activityStmt = $pdo->prepare("
        SELECT activity_type, activity_data, keywords, source, created_at, updated_at
        FROM user_activity
        WHERE customer_id = :customer_id
        ORDER BY updated_at DESC
    ");
    $activityStmt->execute([':customer_id' => $customerId]);
    $activities = $activityStmt->fetchAll();
    
    // Get user interests from customers table
    $userStmt = $pdo->prepare("SELECT interests, activity_data FROM customers WHERE id = :id");
    $userStmt->execute([':id' => $customerId]);
    $userData = $userStmt->fetch();
    
    // Parse activity data
    $parsedActivities = [];
    foreach ($activities as $activity) {
        $activityData = json_decode($activity['activity_data'], true) ?? [];
        $parsedActivities[] = [
            'type' => $activity['activity_type'],
            'source' => $activity['source'],
            'keywords' => $activity['keywords'],
            'data' => $activityData,
            'updated_at' => $activity['updated_at']
        ];
    }
    
    // Group preferences by type
    $groupedPreferences = [];
    foreach ($preferences as $pref) {
        $type = $pref['preference_type'];
        if (!isset($groupedPreferences[$type])) {
            $groupedPreferences[$type] = [];
        }
        $groupedPreferences[$type][] = [
            'value' => $pref['preference_value'],
            'weight' => (float)$pref['weight'],
            'source' => $pref['source']
        ];
    }
    
    // Extract interests
    $interests = [];
    if (!empty($userData['interests'])) {
        $interests = array_map('trim', explode(',', $userData['interests']));
        $interests = array_filter($interests);
    }
    
    echo json_encode([
        'success' => true,
        'preferences' => $groupedPreferences,
        'activities' => $parsedActivities,
        'interests' => array_values($interests),
        'has_preferences' => !empty($preferences),
        'has_activity' => !empty($activities)
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    error_log('User Preferences API Error: ' . $e->getMessage());
    echo json_encode([
        'success' => false,
        'error' => 'Failed to fetch preferences',
        'message' => $e->getMessage()
    ]);
}
