<?php
/**
 * Stripe Webhook Handler
 * Handles Stripe webhook events to update payment status
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once __DIR__ . '/../../config.php';

// Load Stripe PHP library
$autoloadPath = __DIR__ . '/../../vendor/autoload.php';
if (!file_exists($autoloadPath)) {
    http_response_code(500);
    error_log('Stripe library not installed. Please run: composer install');
    exit;
}

require_once $autoloadPath;

use Stripe\Stripe;
use Stripe\Webhook;
use Stripe\Exception\SignatureVerificationException;

// Set Stripe API key
Stripe::setApiKey(STRIPE_SECRET_KEY_TEST);

// Database connection
try {
    $pdo = new PDO(
        "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
        DB_USER,
        DB_PASS,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]
    );
} catch (PDOException $e) {
    http_response_code(500);
    error_log('Database connection failed: ' . $e->getMessage());
    exit;
}

// Get webhook payload
$payload = @file_get_contents('php://input');
$sig_header = $_SERVER['HTTP_STRIPE_SIGNATURE'] ?? '';

// Verify webhook signature (if webhook secret is configured)
if (STRIPE_WEBHOOK_SECRET) {
    try {
        $event = Webhook::constructEvent($payload, $sig_header, STRIPE_WEBHOOK_SECRET);
    } catch (SignatureVerificationException $e) {
        http_response_code(400);
        error_log('Webhook signature verification failed: ' . $e->getMessage());
        exit;
    }
} else {
    // If no webhook secret, decode JSON directly (for testing)
    $event = json_decode($payload, true);
}

// Handle the event
try {
    switch ($event['type']) {
        case 'checkout.session.completed':
            $session = $event['data']['object'];
            handleCheckoutSessionCompleted($pdo, $session);
            break;
            
        case 'payment_intent.succeeded':
            $paymentIntent = $event['data']['object'];
            handlePaymentIntentSucceeded($pdo, $paymentIntent);
            break;
            
        case 'payment_intent.payment_failed':
            $paymentIntent = $event['data']['object'];
            handlePaymentIntentFailed($pdo, $paymentIntent);
            break;
            
        default:
            error_log('Unhandled event type: ' . $event['type']);
    }
    
    http_response_code(200);
    echo json_encode(['received' => true]);
    
} catch (Exception $e) {
    error_log('Webhook handler error: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

/**
 * Handle checkout session completed
 */
function handleCheckoutSessionCompleted($pdo, $session) {
    $checkoutSessionId = $session['id'];
    $paymentStatus = $session['payment_status'];
    $customerEmail = $session['customer_details']['email'] ?? null;
    $customerName = $session['customer_details']['name'] ?? null;
    
    // Update payment record
    $stmt = $pdo->prepare("
        UPDATE payments 
        SET payment_status = :status,
            customer_email = COALESCE(:customer_email, customer_email),
            customer_name = COALESCE(:customer_name, customer_name),
            stripe_customer_id = :customer_id,
            updated_at = NOW()
        WHERE stripe_checkout_session_id = :session_id
    ");
    
    $stmt->execute([
        ':status' => $paymentStatus === 'paid' ? 'completed' : $paymentStatus,
        ':customer_email' => $customerEmail,
        ':customer_name' => $customerName,
        ':customer_id' => $session['customer'] ?? null,
        ':session_id' => $checkoutSessionId,
    ]);
}

/**
 * Handle payment intent succeeded
 */
function handlePaymentIntentSucceeded($pdo, $paymentIntent) {
    $paymentIntentId = $paymentIntent['id'];
    $customerEmail = $paymentIntent['receipt_email'] ?? null;
    
    // Update payment record
    $stmt = $pdo->prepare("
        UPDATE payments 
        SET stripe_payment_intent_id = :payment_intent_id,
            payment_status = 'completed',
            payment_method = :payment_method,
            customer_email = COALESCE(:customer_email, customer_email),
            updated_at = NOW()
        WHERE stripe_checkout_session_id IN (
            SELECT id FROM stripe_checkout_sessions WHERE payment_intent = :payment_intent_id
        ) OR stripe_payment_intent_id = :payment_intent_id
    ");
    
    $stmt->execute([
        ':payment_intent_id' => $paymentIntentId,
        ':payment_method' => $paymentIntent['payment_method_types'][0] ?? null,
        ':customer_email' => $customerEmail,
    ]);
}

/**
 * Handle payment intent failed
 */
function handlePaymentIntentFailed($pdo, $paymentIntent) {
    $paymentIntentId = $paymentIntent['id'];
    
    // Update payment record
    $stmt = $pdo->prepare("
        UPDATE payments 
        SET stripe_payment_intent_id = :payment_intent_id,
            payment_status = 'failed',
            updated_at = NOW()
        WHERE stripe_checkout_session_id IN (
            SELECT id FROM stripe_checkout_sessions WHERE payment_intent = :payment_intent_id
        ) OR stripe_payment_intent_id = :payment_intent_id
    ");
    
    $stmt->execute([
        ':payment_intent_id' => $paymentIntentId,
    ]);
}

