<?php
/**
 * Verify Payment Status API
 * Verifies payment status after Stripe checkout
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: Content-Type');

require_once __DIR__ . '/../../config.php';

// Load Stripe PHP library
$autoloadPath = __DIR__ . '/../../vendor/autoload.php';
if (!file_exists($autoloadPath)) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Stripe library not installed',
        'message' => 'Please run: composer install'
    ]);
    exit;
}

require_once $autoloadPath;

use Stripe\Stripe;
use Stripe\Checkout\Session;
use Stripe\Exception\ApiErrorException;

// Set Stripe API key
Stripe::setApiKey(STRIPE_SECRET_KEY_TEST);

// Database connection
try {
    $pdo = new PDO(
        "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
        DB_USER,
        DB_PASS,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]
    );
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Database connection failed'
    ]);
    exit;
}

// Get session ID from query parameters
$sessionId = isset($_GET['session_id']) ? trim($_GET['session_id']) : null;

if (!$sessionId) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'Session ID is required'
    ]);
    exit;
}

try {
    // Retrieve checkout session from Stripe
    $checkoutSession = Session::retrieve($sessionId);
    
    // Get payment record from database
    $stmt = $pdo->prepare("
        SELECT * FROM payments 
        WHERE stripe_checkout_session_id = :session_id 
        LIMIT 1
    ");
    $stmt->execute([':session_id' => $sessionId]);
    $payment = $stmt->fetch();
    
    if (!$payment) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'error' => 'Payment record not found'
        ]);
        exit;
    }
    
    // Update payment status based on Stripe session status
    $paymentStatus = 'pending';
    if ($checkoutSession->payment_status === 'paid') {
        $paymentStatus = 'completed';
    } elseif ($checkoutSession->payment_status === 'unpaid') {
        $paymentStatus = 'pending';
    }
    
    // Update database if status changed
    if ($payment['payment_status'] !== $paymentStatus) {
        $updateStmt = $pdo->prepare("
            UPDATE payments 
            SET payment_status = :status,
                stripe_payment_intent_id = :payment_intent_id,
                stripe_customer_id = :customer_id,
                updated_at = NOW()
            WHERE id = :id
        ");
        
        $updateStmt->execute([
            ':status' => $paymentStatus,
            ':payment_intent_id' => $checkoutSession->payment_intent ?? null,
            ':customer_id' => $checkoutSession->customer ?? null,
            ':id' => $payment['id'],
        ]);
        
        $payment['payment_status'] = $paymentStatus;
    }
    
    echo json_encode([
        'success' => true,
        'payment' => [
            'id' => $payment['id'],
            'event_id' => $payment['event_id'],
            'event_name' => $payment['event_name'],
            'amount' => $payment['amount'],
            'currency' => $payment['currency'],
            'payment_status' => $payment['payment_status'],
            'ticket_quantity' => $payment['ticket_quantity'],
            'created_at' => $payment['created_at'],
        ],
        'stripe_session' => [
            'payment_status' => $checkoutSession->payment_status,
            'status' => $checkoutSession->status,
        ]
    ]);
    
} catch (ApiErrorException $e) {
    error_log('Stripe API Error: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Failed to verify payment',
        'message' => $e->getMessage()
    ]);
} catch (Exception $e) {
    error_log('Error: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Unexpected error',
        'message' => $e->getMessage()
    ]);
}

