<?php
/**
 * Create Stripe Checkout Session API
 * Creates a Stripe checkout session for event ticket payment
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

require_once __DIR__ . '/../../config.php';

// Load Stripe PHP library
$autoloadPath = __DIR__ . '/../../vendor/autoload.php';
if (!file_exists($autoloadPath)) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Stripe library not installed',
        'message' => 'Please run: composer install'
    ]);
    exit;
}

require_once $autoloadPath;

use Stripe\Stripe;
use Stripe\Checkout\Session;
use Stripe\Exception\ApiErrorException;

// Set Stripe API key
Stripe::setApiKey(STRIPE_SECRET_KEY_TEST);

// Database connection
try {
    $pdo = new PDO(
        "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
        DB_USER,
        DB_PASS,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]
    );
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Database connection failed',
        'message' => $e->getMessage()
    ]);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate input
$eventId = isset($input['event_id']) ? trim($input['event_id']) : null;
$quantity = isset($input['quantity']) ? intval($input['quantity']) : 1;
$customerEmail = isset($input['customer_email']) ? trim($input['customer_email']) : null;
$customerName = isset($input['customer_name']) ? trim($input['customer_name']) : null;

if (!$eventId) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'Event ID is required'
    ]);
    exit;
}

if ($quantity < 1) {
    $quantity = 1;
}

// Fetch event details
try {
    $stmt = $pdo->prepare("SELECT id, name, pricing FROM events WHERE id = :id LIMIT 1");
    $stmt->execute([':id' => $eventId]);
    $event = $stmt->fetch();
    
    if (!$event) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'error' => 'Event not found'
        ]);
        exit;
    }
    
    // Parse pricing - handle different formats like "Free", "$50", "50 USD", etc.
    $amount = 0;
    $currency = 'USD';
    
    $pricing = $event['pricing'];
    if ($pricing && strtolower($pricing) !== 'free' && strtolower($pricing) !== 'tba') {
        // Extract numeric value from pricing string
        preg_match('/[\d.]+/', $pricing, $matches);
        if (!empty($matches)) {
            $amount = floatval($matches[0]) * 100; // Convert to cents
        }
    }
    
    // If amount is 0, return error (free events don't need payment)
    if ($amount <= 0) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'error' => 'This event is free or pricing is not available',
            'message' => 'Payment is not required for this event'
        ]);
        exit;
    }
    
    // Calculate total amount
    $totalAmount = intval($amount * $quantity);
    
    // Create success and cancel URLs
    $successUrl = BASE_URL . '/event-detail.html?id=' . urlencode($eventId) . '&payment=success&session_id={CHECKOUT_SESSION_ID}';
    $cancelUrl = BASE_URL . '/event-detail.html?id=' . urlencode($eventId) . '&payment=cancelled';
    
    // Create Stripe Checkout Session
    try {
        $sessionParams = [
            'payment_method_types' => ['card'],
            'line_items' => [[
                'price_data' => [
                    'currency' => strtolower($currency),
                    'product_data' => [
                        'name' => $event['name'] . ' - Ticket' . ($quantity > 1 ? 's' : ''),
                        'description' => 'Event Ticket Purchase',
                    ],
                    'unit_amount' => intval($amount),
                ],
                'quantity' => $quantity,
            ]],
            'mode' => 'payment',
            'success_url' => $successUrl,
            'cancel_url' => $cancelUrl,
            'metadata' => [
                'event_id' => $eventId,
                'event_name' => $event['name'],
                'quantity' => $quantity,
            ],
        ];
        
        // Add customer email if provided
        if ($customerEmail && filter_var($customerEmail, FILTER_VALIDATE_EMAIL)) {
            $sessionParams['customer_email'] = $customerEmail;
        }
        
        $checkoutSession = Session::create($sessionParams);
        
        // Save payment record to database
        $stmt = $pdo->prepare("
            INSERT INTO payments (
                event_id, event_name, stripe_checkout_session_id,
                amount, currency, payment_status, ticket_quantity,
                customer_email, customer_name, created_at
            ) VALUES (
                :event_id, :event_name, :stripe_checkout_session_id,
                :amount, :currency, 'pending', :ticket_quantity,
                :customer_email, :customer_name, NOW()
            )
        ");
        
        $stmt->execute([
            ':event_id' => $eventId,
            ':event_name' => $event['name'],
            ':stripe_checkout_session_id' => $checkoutSession->id,
            ':amount' => $totalAmount / 100, // Store in dollars
            ':currency' => $currency,
            ':ticket_quantity' => $quantity,
            ':customer_email' => $customerEmail ?: null,
            ':customer_name' => $customerName ?: null,
        ]);
        
        echo json_encode([
            'success' => true,
            'session_id' => $checkoutSession->id,
            'checkout_url' => $checkoutSession->url,
        ]);
        
    } catch (ApiErrorException $e) {
        error_log('Stripe API Error: ' . $e->getMessage());
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => 'Payment session creation failed',
            'message' => $e->getMessage()
        ]);
    }
    
} catch (PDOException $e) {
    error_log('Database Error: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Database query failed',
        'message' => 'Unable to process payment. Please try again later.'
    ]);
} catch (Exception $e) {
    error_log('Error: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Unexpected error',
        'message' => 'An error occurred. Please try again later.'
    ]);
}

