<?php
/**
 * Events API Endpoint
 * Returns events from the database in JSON format
 * Uses personalized recommendations if user is logged in, otherwise shows all events
 */

// Error reporting (disable display, enable logging)
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: Content-Type');

// Start session for user authentication check
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Include recommendation system
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/recommendations.php';

// Get query parameters
$eventType = isset($_GET['type']) ? trim($_GET['type']) : null;
$country = isset($_GET['country']) ? trim($_GET['country']) : null;
$limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 100;

try {
    // Get personalized events (will fallback to all events if no preferences)
    $events = getPersonalizedEvents($eventType, $country, $limit);
    
    // Format events for frontend (remove relevance_score if present)
    $formattedEvents = array_map(function($event) {
        $formatted = [
            'id' => $event['id'],
            'name' => $event['name'],
            'start_date' => $event['start_date'],
            'end_date' => $event['end_date'],
            'attendees_range' => $event['attendees_range'],
            'venue' => $event['venue'],
            'city' => $event['city'],
            'country' => $event['country'],
            'ticket_available' => (bool)($event['ticket_available'] ?? false),
            'pricing' => $event['pricing'] ?? null,
            'description' => $event['description'] ?? null,
            'attendee_profiles' => $event['attendee_profiles'] ?? null,
            'sponsors' => $event['sponsors'] ?? null,
            'event_type' => $event['event_type'] ?? null,
            'website_url' => $event['website_url'] ?? null,
        ];
        
        // Include relevance score if available (for debugging/testing)
        if (isset($event['relevance_score'])) {
            $formatted['relevance_score'] = (float)$event['relevance_score'];
        }
        
        return $formatted;
    }, $events);
    
    echo json_encode([
        'success' => true,
        'events' => $formattedEvents,
        'count' => count($formattedEvents),
        'personalized' => isLoggedIn() && !empty(getUserPreferences($_SESSION['user_id'] ?? 0))
    ]);
    
} catch (PDOException $e) {
    http_response_code(500);
    error_log('Events API Error: ' . $e->getMessage());
    echo json_encode([
        'success' => false,
        'error' => 'Database query failed',
        'message' => 'Unable to fetch events. Please try again later.'
    ]);
} catch (Exception $e) {
    http_response_code(500);
    error_log('Events API Error: ' . $e->getMessage());
    echo json_encode([
        'success' => false,
        'error' => 'Unexpected error',
        'message' => 'An error occurred. Please try again later.'
    ]);
}
