<?php
/**
 * Google OAuth Callback - Handle OAuth response
 */

require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/user_activity.php';

// Check for errors
if (isset($_GET['error'])) {
    header('Location: ../../login.html?error=' . urlencode('Authentication failed: ' . $_GET['error']));
    exit;
}

// Verify state token
if (!isset($_GET['state']) || !isset($_SESSION['oauth_state']) || $_GET['state'] !== $_SESSION['oauth_state']) {
    header('Location: ../../login.html?error=' . urlencode('Invalid state token. Please try again.'));
    exit;
}

// Check for authorization code
if (!isset($_GET['code'])) {
    header('Location: ../../login.html?error=' . urlencode('Authorization code not received.'));
    exit;
}

$code = $_GET['code'];

try {
    // Exchange authorization code for access token
    $tokenUrl = 'https://oauth2.googleapis.com/token';
    $tokenData = [
        'code' => $code,
        'client_id' => GOOGLE_CLIENT_ID,
        'client_secret' => GOOGLE_CLIENT_SECRET,
        'redirect_uri' => GOOGLE_REDIRECT_URI,
        'grant_type' => 'authorization_code'
    ];
    
    $ch = curl_init($tokenUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($tokenData));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/x-www-form-urlencoded']);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200) {
        throw new Exception('Failed to exchange authorization code for token');
    }
    
    $tokenResponse = json_decode($response, true);
    
    if (!isset($tokenResponse['access_token'])) {
        throw new Exception('Access token not received');
    }
    
    $accessToken = $tokenResponse['access_token'];
    $refreshToken = $tokenResponse['refresh_token'] ?? null;
    
    // Get user info from Google
    $userInfoUrl = 'https://www.googleapis.com/oauth2/v2/userinfo?access_token=' . urlencode($accessToken);
    
    $ch = curl_init($userInfoUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Authorization: Bearer ' . $accessToken]);
    
    $userResponse = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200) {
        throw new Exception('Failed to get user information from Google');
    }
    
    $userData = json_decode($userResponse, true);
    
    // Add tokens to user data
    $userData['access_token'] = $accessToken;
    $userData['refresh_token'] = $refreshToken;
    
    // Create or update user
    $userId = createOrUpdateUser('google', $userData);
    
    // Fetch and store user activity (async-like, don't block on errors)
    try {
        fetchGoogleActivity($accessToken, $userId);
    } catch (Exception $e) {
        error_log('Error fetching Google activity: ' . $e->getMessage());
        // Continue with login even if activity fetch fails
    }
    
    // Get full user data
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT * FROM customers WHERE id = :id");
    $stmt->execute([':id' => $userId]);
    $fullUserData = $stmt->fetch();
    
    // Login user
    loginUser($userId, $fullUserData);
    
    // Clear OAuth state
    unset($_SESSION['oauth_state']);
    unset($_SESSION['oauth_provider']);
    
    // Redirect to home
    header('Location: ../../index.html');
    exit;
    
} catch (Exception $e) {
    error_log('Google OAuth Error: ' . $e->getMessage());
    header('Location: ../../login.html?error=' . urlencode('Authentication failed. Please try again.'));
    exit;
}
