<?php
$pageTitle = 'Admin Users Management';
require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/includes/db.php';
requirePermission('users.view');

$pdo = getDB();

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$perPage = ITEMS_PER_PAGE;
$offset = ($page - 1) * $perPage;

// Search
$search = $_GET['search'] ?? '';

$where = ['1=1'];
$params = [];

if ($search) {
    $where[] = "(username LIKE :search OR email LIKE :search OR full_name LIKE :search)";
    $params[':search'] = '%' . $search . '%';
}

$whereClause = implode(' AND ', $where);

// Get total count
$countStmt = $pdo->prepare("SELECT COUNT(*) as total FROM admin_users WHERE $whereClause");
$countStmt->execute($params);
$totalItems = $countStmt->fetch()['total'];
$totalPages = ceil($totalItems / $perPage);

// Get users
$sql = "SELECT a.*, r.name as role_name FROM admin_users a LEFT JOIN roles r ON a.role_id = r.id WHERE $whereClause ORDER BY a.created_at DESC LIMIT :limit OFFSET :offset";
$stmt = $pdo->prepare($sql);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$users = $stmt->fetchAll();
?>

<div class="page-header">
    <h1 class="page-title">Admin Users Management</h1>
    <p class="page-subtitle">Manage admin panel users and access</p>
</div>

<?php if (hasPermission('users.create')): ?>
<div class="card">
    <a href="user-add.php" class="btn btn-primary">+ Add New Admin User</a>
</div>
<?php endif; ?>

<!-- Search -->
<div class="card">
    <form method="GET" action="" style="display: flex; gap: 1rem; align-items: end;">
        <div class="form-group" style="flex: 1;">
            <label class="form-label">Search</label>
            <input type="text" name="search" class="form-control" placeholder="Search by username, email, or name..." value="<?php echo htmlspecialchars($search); ?>">
        </div>
        <div class="form-group">
            <button type="submit" class="btn btn-primary">Search</button>
            <a href="users.php" class="btn btn-secondary">Reset</a>
        </div>
    </form>
</div>

<!-- Users Table -->
<div class="card">
    <div class="card-header">
        <h2 class="card-title">Admin Users (<?php echo number_format($totalItems); ?>)</h2>
    </div>
    
    <div class="table-container">
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Username</th>
                    <th>Full Name</th>
                    <th>Email</th>
                    <th>Role</th>
                    <th>Status</th>
                    <th>Last Login</th>
                    <th>Created</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($users)): ?>
                    <tr>
                        <td colspan="9" style="text-align: center; color: var(--text-medium); padding: 2rem;">No users found</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($users as $user): ?>
                        <tr>
                            <td><?php echo $user['id']; ?></td>
                            <td><strong><?php echo htmlspecialchars($user['username']); ?></strong></td>
                            <td><?php echo htmlspecialchars($user['full_name'] ?? 'N/A'); ?></td>
                            <td><?php echo htmlspecialchars($user['email']); ?></td>
                            <td><span class="badge badge-info"><?php echo htmlspecialchars($user['role_name'] ?? 'No Role'); ?></span></td>
                            <td>
                                <?php if ($user['is_active']): ?>
                                    <span class="badge badge-success">Active</span>
                                <?php else: ?>
                                    <span class="badge badge-danger">Inactive</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo $user['last_login'] ? date('M d, Y H:i', strtotime($user['last_login'])) : 'Never'; ?></td>
                            <td><?php echo date('M d, Y', strtotime($user['created_at'])); ?></td>
                            <td>
                                <div style="display: flex; gap: 0.5rem;">
                                    <?php if (hasPermission('users.edit')): ?>
                                        <a href="user-edit.php?id=<?php echo $user['id']; ?>" class="btn btn-secondary btn-sm">Edit</a>
                                    <?php endif; ?>
                                    <?php if (hasPermission('users.delete') && $user['id'] != $_SESSION['admin_id']): ?>
                                        <a href="user-delete.php?id=<?php echo $user['id']; ?>" class="btn btn-danger btn-sm" onclick="return confirm('Are you sure?')">Delete</a>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Pagination -->
    <?php if ($totalPages > 1): ?>
        <div class="pagination">
            <?php if ($page > 1): ?>
                <a href="?page=<?php echo $page - 1; ?>&search=<?php echo urlencode($search); ?>">Previous</a>
            <?php endif; ?>
            
            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                <a href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>" 
                   class="<?php echo $i === $page ? 'active' : ''; ?>">
                    <?php echo $i; ?>
                </a>
            <?php endfor; ?>
            
            <?php if ($page < $totalPages): ?>
                <a href="?page=<?php echo $page + 1; ?>&search=<?php echo urlencode($search); ?>">Next</a>
            <?php endif; ?>
        </div>
    <?php endif; ?>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
