<?php
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/includes/db.php';

// Redirect if already logged in
if (isset($_SESSION['admin_id'])) {
    header('Location: ' . ADMIN_BASE_URL . '/index.php');
    exit;
}

$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';
    
    if (!empty($username) && !empty($password)) {
        try {
            $pdo = getDB();
            $stmt = $pdo->prepare("
                SELECT a.*, r.name as role_name, r.permissions 
                FROM admin_users a 
                LEFT JOIN roles r ON a.role_id = r.id 
                WHERE a.username = :username AND a.is_active = 1
            ");
            $stmt->execute([':username' => $username]);
            $admin = $stmt->fetch();
            
            if ($admin && password_verify($password, $admin['password_hash'])) {
                $_SESSION['admin_id'] = $admin['id'];
                $_SESSION['admin_username'] = $admin['username'];
                $_SESSION['admin_email'] = $admin['email'];
                $_SESSION['admin_role'] = $admin['role_name'];
                
                // Update last login
                $stmt = $pdo->prepare("UPDATE admin_users SET last_login = NOW() WHERE id = :id");
                $stmt->execute([':id' => $admin['id']]);
                
                header('Location: ' . ADMIN_BASE_URL . '/index.php');
                exit;
            } else {
                $error = 'Invalid username or password';
            }
        } catch (Exception $e) {
            $error = 'Login failed. Please try again.';
            error_log('Admin login error: ' . $e->getMessage());
        }
    } else {
        $error = 'Please enter both username and password';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Login | beLIVE</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="<?php echo ADMIN_BASE_URL; ?>/css/admin.css">
    <style>
        .login-wrapper {
            min-height: 100vh;
            display: flex;
        }
        
        .login-left {
            flex: 1;
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
            display: flex;
            flex-direction: column;
            justify-content: center;
            padding: 4rem;
            color: white;
            position: relative;
            overflow: hidden;
        }
        
        .login-left::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg width="100" height="100" xmlns="http://www.w3.org/2000/svg"><defs><pattern id="grid" width="100" height="100" patternUnits="userSpaceOnUse"><path d="M 100 0 L 0 0 0 100" fill="none" stroke="rgba(255,255,255,0.05)" stroke-width="1"/></pattern></defs><rect width="100" height="100" fill="url(%23grid)"/></svg>');
            opacity: 0.3;
        }
        
        .login-left-content {
            position: relative;
            z-index: 1;
            max-width: 500px;
        }
        
        .login-logo {
            margin-bottom: 3rem;
        }
        
        .login-logo img {
            height: 60px;
            width: auto;
            max-width: 100%;
            object-fit: contain;
            filter: brightness(0) invert(1);
        }
        
        .login-left h1 {
            font-size: 3rem;
            font-weight: 800;
            margin-bottom: 1.5rem;
            line-height: 1.2;
            letter-spacing: -0.03em;
        }
        
        .login-left p {
            font-size: 1.25rem;
            line-height: 1.8;
            margin-bottom: 2rem;
            opacity: 0.95;
        }
        
        .login-features {
            list-style: none;
            padding: 0;
            margin-top: 2rem;
        }
        
        .login-features li {
            display: flex;
            align-items: center;
            gap: 1rem;
            margin-bottom: 1.25rem;
            font-size: 1rem;
        }
        
        .login-features li::before {
            content: '✓';
            width: 28px;
            height: 28px;
            background: rgba(255, 255, 255, 0.2);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            flex-shrink: 0;
        }
        
        .login-right {
            flex: 1;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 3rem;
            background: linear-gradient(135deg, #fef2f2 0%, #fff7f7 50%, #fef2f2 100%);
        }
        
        .login-card {
            background: white;
            border-radius: 20px;
            padding: 3.5rem;
            max-width: 450px;
            width: 100%;
            box-shadow: 0 10px 40px rgba(220, 38, 38, 0.12);
            border: 1px solid rgba(220, 38, 38, 0.1);
        }
        
        .login-header {
            text-align: center;
            margin-bottom: 2.5rem;
        }
        
        .login-header h2 {
            color: var(--text-dark);
            font-size: 1.75rem;
            font-weight: 800;
            margin-bottom: 0.75rem;
            letter-spacing: -0.02em;
        }
        
        .login-header p {
            color: var(--text-medium);
            font-size: 0.95rem;
        }
        
        .form-group {
            margin-bottom: 1.5rem;
        }
        
        .form-label {
            display: block;
            margin-bottom: 0.75rem;
            font-weight: 600;
            color: var(--text-dark);
            font-size: 0.9rem;
        }
        
        .form-control {
            width: 100%;
            padding: 0.875rem 1rem;
            border: 2px solid var(--border-color);
            border-radius: 10px;
            font-size: 1rem;
            transition: all 0.3s;
            background: #fafafa;
        }
        
        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            background: white;
            box-shadow: 0 0 0 4px rgba(220, 38, 38, 0.1);
        }
        
        .btn-primary {
            width: 100%;
            padding: 0.875rem;
            font-size: 1rem;
            font-weight: 600;
            margin-top: 0.5rem;
        }
        
        .error-message {
            background: rgba(231, 76, 60, 0.1);
            color: var(--danger);
            padding: 1rem 1.25rem;
            border-radius: 10px;
            margin-bottom: 1.5rem;
            border-left: 4px solid var(--danger);
            font-size: 0.9rem;
        }
        
        .back-link {
            text-align: center;
            margin-top: 2rem;
            padding-top: 2rem;
            border-top: 1px solid var(--border-color);
        }
        
        .back-link a {
            color: var(--text-medium);
            text-decoration: none;
            font-size: 0.9rem;
            transition: color 0.3s;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .back-link a:hover {
            color: var(--primary);
        }
        
        @media (max-width: 968px) {
            .login-wrapper {
                flex-direction: column;
            }
            
            .login-left {
                padding: 2rem;
                min-height: 40vh;
            }
            
            .login-left h1 {
                font-size: 2rem;
            }
            
            .login-left p {
                font-size: 1rem;
            }
            
            .login-right {
                padding: 2rem;
            }
            
            .login-logo img {
                height: 50px;
            }
        }
        
        @media (max-width: 480px) {
            .login-logo img {
                height: 40px;
            }
        }
    </style>
</head>
<body>
    <div class="login-wrapper">
        <!-- Left Side - Marketing Content -->
        <div class="login-left">
            <div class="login-left-content">
                <div class="login-logo">
                    <img src="https://belive.mobi/images/belive.png" alt="beLIVE">
                </div>
                <h1>Welcome to<br>beLIVE Admin</h1>
                <p>Manage your events, customers, and platform with powerful tools designed for efficiency and control.</p>
                
                <ul class="login-features">
                    <li>Complete event management and moderation</li>
                    <li>Customer insights and analytics</li>
                    <li>Role-based access control</li>
                    <li>Real-time dashboard updates</li>
                </ul>
            </div>
        </div>
        
        <!-- Right Side - Login Form -->
        <div class="login-right">
            <div class="login-card">
                <div class="login-header">
                    <h2>Sign In</h2>
                    <p>Enter your credentials to access the admin panel</p>
                </div>
                
                <?php if ($error): ?>
                    <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
                <?php endif; ?>
                
                <form method="POST" action="">
                    <div class="form-group">
                        <label class="form-label" for="username">Username</label>
                        <input type="text" id="username" name="username" class="form-control" placeholder="Enter your username" required autofocus>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="password">Password</label>
                        <input type="password" id="password" name="password" class="form-control" placeholder="Enter your password" required>
                    </div>
                    
                    <button type="submit" class="btn btn-primary">Sign In</button>
                </form>
                
                <div class="back-link">
                    <a href="<?php echo str_replace('/admin', '', BASE_URL); ?>">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M19 12H5M12 19l-7-7 7-7"/>
                        </svg>
                        Back to Website
                    </a>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
