<?php
/**
 * Admin Authentication Functions
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/db.php';

/**
 * Check if admin is logged in
 */
function isAdminLoggedIn() {
    return isset($_SESSION['admin_id']) && !empty($_SESSION['admin_id']);
}

/**
 * Require admin login - redirect if not logged in
 */
function requireAdminLogin() {
    if (!isAdminLoggedIn()) {
        header('Location: ' . ADMIN_BASE_URL . '/login.php');
        exit;
    }
}

/**
 * Get current admin user
 */
function getCurrentAdmin() {
    if (!isAdminLoggedIn()) {
        return null;
    }
    
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("
            SELECT a.*, r.name as role_name, r.permissions 
            FROM admin_users a 
            LEFT JOIN roles r ON a.role_id = r.id 
            WHERE a.id = :id AND a.is_active = 1
        ");
        $stmt->execute([':id' => $_SESSION['admin_id']]);
        $admin = $stmt->fetch();
        
        if ($admin && $admin['permissions']) {
            $admin['permissions'] = json_decode($admin['permissions'], true) ?? [];
        }
        
        return $admin;
    } catch (Exception $e) {
        error_log('Error getting current admin: ' . $e->getMessage());
        return null;
    }
}

/**
 * Check if admin has permission
 */
function hasPermission($permission) {
    $admin = getCurrentAdmin();
    if (!$admin) {
        return false;
    }
    
    // Super admin has all permissions
    if (in_array('*', $admin['permissions'] ?? [])) {
        return true;
    }
    
    // Check exact permission
    if (in_array($permission, $admin['permissions'] ?? [])) {
        return true;
    }
    
    // Check wildcard permissions (e.g., "events.*" matches "events.edit")
    foreach ($admin['permissions'] ?? [] as $perm) {
        if (strpos($perm, '*') !== false) {
            $pattern = str_replace('*', '.*', preg_quote($perm, '/'));
            if (preg_match('/^' . $pattern . '$/', $permission)) {
                return true;
            }
        }
    }
    
    return false;
}

/**
 * Require specific permission
 */
function requirePermission($permission) {
    requireAdminLogin();
    
    if (!hasPermission($permission)) {
        header('HTTP/1.1 403 Forbidden');
        die('Access denied. You do not have permission to perform this action.');
    }
}

/**
 * Login admin
 */
function loginAdmin($adminId, $adminData = null) {
    $_SESSION['admin_id'] = $adminId;
    $_SESSION['admin_username'] = $adminData['username'] ?? null;
    $_SESSION['admin_email'] = $adminData['email'] ?? null;
    $_SESSION['admin_role'] = $adminData['role_name'] ?? null;
    
    // Update last login
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("UPDATE admin_users SET last_login = NOW() WHERE id = :id");
        $stmt->execute([':id' => $adminId]);
    } catch (Exception $e) {
        error_log('Error updating admin last login: ' . $e->getMessage());
    }
}

/**
 * Logout admin
 */
function logoutAdmin() {
    $_SESSION = array();
    
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }
    
    session_destroy();
}

/**
 * Verify admin password
 */
function verifyAdminPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Hash admin password
 */
function hashAdminPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}
