<?php
$pageTitle = 'Events Management';
require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/includes/db.php';
requirePermission('events.view');

$pdo = getDB();

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$perPage = ITEMS_PER_PAGE;
$offset = ($page - 1) * $perPage;

// Search and filters
$search = $_GET['search'] ?? '';
$eventType = $_GET['event_type'] ?? '';

$where = ['1=1'];
$params = [];

if ($search) {
    $where[] = "(name LIKE :search OR description LIKE :search OR city LIKE :search OR country LIKE :search)";
    $params[':search'] = '%' . $search . '%';
}

if ($eventType) {
    $where[] = "event_type = :event_type";
    $params[':event_type'] = $eventType;
}

$whereClause = implode(' AND ', $where);

// Get total count
$countStmt = $pdo->prepare("SELECT COUNT(*) as total FROM events WHERE $whereClause");
$countStmt->execute($params);
$totalItems = $countStmt->fetch()['total'];
$totalPages = ceil($totalItems / $perPage);

// Get events
$sql = "SELECT * FROM events WHERE $whereClause ORDER BY created_at DESC LIMIT :limit OFFSET :offset";
$stmt = $pdo->prepare($sql);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$events = $stmt->fetchAll();

// Get event types for filter
$typeStmt = $pdo->query("SELECT DISTINCT event_type FROM events WHERE event_type IS NOT NULL AND event_type != '' ORDER BY event_type");
$eventTypes = $typeStmt->fetchAll(PDO::FETCH_COLUMN);
?>

<div class="page-header">
    <h1 class="page-title">Events Management</h1>
    <p class="page-subtitle">Manage all events in the system</p>
</div>

<?php if (isset($_SESSION['success_message'])): ?>
    <div class="alert alert-success" style="background: rgba(39, 174, 96, 0.1); color: var(--success); padding: 1rem 1.25rem; border-radius: 10px; margin-bottom: 1.5rem; border-left: 4px solid var(--success);">
        <?php echo htmlspecialchars($_SESSION['success_message']); unset($_SESSION['success_message']); ?>
    </div>
<?php endif; ?>

<?php if (hasPermission('events.create')): ?>
<div class="card">
    <a href="event-add.php" class="btn btn-primary">+ Add New Event</a>
</div>
<?php endif; ?>

<!-- Search and Filters -->
<div class="card">
    <form method="GET" action="" style="display: flex; gap: 1rem; flex-wrap: wrap; align-items: end;">
        <div class="form-group" style="flex: 1; min-width: 200px;">
            <label class="form-label">Search</label>
            <input type="text" name="search" class="form-control" placeholder="Search events..." value="<?php echo htmlspecialchars($search); ?>">
        </div>
        
        <div class="form-group" style="min-width: 150px;">
            <label class="form-label">Event Type</label>
            <select name="event_type" class="form-control">
                <option value="">All Types</option>
                <?php foreach ($eventTypes as $type): ?>
                    <option value="<?php echo htmlspecialchars($type); ?>" <?php echo $eventType === $type ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars(ucfirst($type)); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        
        <div class="form-group">
            <button type="submit" class="btn btn-primary">Filter</button>
            <a href="events.php" class="btn btn-secondary">Reset</a>
        </div>
    </form>
</div>

<!-- Events Table -->
<div class="card">
    <div class="card-header">
        <h2 class="card-title">Events (<?php echo number_format($totalItems); ?>)</h2>
    </div>
    
    <div class="table-container">
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Event Name</th>
                    <th>Type</th>
                    <th>Location</th>
                    <th>Start Date</th>
                    <th>Attendees</th>
                    <th>Created</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($events)): ?>
                    <tr>
                        <td colspan="8" style="text-align: center; color: var(--text-medium); padding: 2rem;">No events found</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($events as $event): ?>
                        <tr>
                            <td><?php echo htmlspecialchars(substr($event['id'], 0, 20)); ?>...</td>
                            <td>
                                <strong><?php echo htmlspecialchars($event['name']); ?></strong>
                                <?php if ($event['description']): ?>
                                    <br><small style="color: var(--text-medium);"><?php echo htmlspecialchars(substr($event['description'], 0, 60)); ?>...</small>
                                <?php endif; ?>
                            </td>
                            <td><span class="badge badge-info"><?php echo htmlspecialchars($event['event_type'] ?? 'N/A'); ?></span></td>
                            <td>
                                <?php 
                                $location = [];
                                if ($event['city']) $location[] = $event['city'];
                                if ($event['country']) $location[] = $event['country'];
                                echo htmlspecialchars(implode(', ', $location) ?: 'TBA');
                                ?>
                            </td>
                            <td><?php echo $event['start_date'] ? date('M d, Y', strtotime($event['start_date'])) : 'TBA'; ?></td>
                            <td><?php echo htmlspecialchars($event['attendees_range'] ?? 'N/A'); ?></td>
                            <td><?php echo date('M d, Y', strtotime($event['created_at'])); ?></td>
                            <td>
                                <div style="display: flex; gap: 0.5rem;">
                                    <?php if (hasPermission('events.edit')): ?>
                                        <a href="event-edit.php?id=<?php echo urlencode($event['id']); ?>" class="btn btn-secondary btn-sm">Edit</a>
                                    <?php endif; ?>
                                    <?php if (hasPermission('events.delete')): ?>
                                        <a href="event-delete.php?id=<?php echo urlencode($event['id']); ?>" class="btn btn-danger btn-sm" onclick="return confirm('Are you sure?')">Delete</a>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Pagination -->
    <?php if ($totalPages > 1): ?>
        <div class="pagination">
            <?php if ($page > 1): ?>
                <a href="?page=<?php echo $page - 1; ?>&search=<?php echo urlencode($search); ?>&event_type=<?php echo urlencode($eventType); ?>">Previous</a>
            <?php endif; ?>
            
            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                <a href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&event_type=<?php echo urlencode($eventType); ?>" 
                   class="<?php echo $i === $page ? 'active' : ''; ?>">
                    <?php echo $i; ?>
                </a>
            <?php endfor; ?>
            
            <?php if ($page < $totalPages): ?>
                <a href="?page=<?php echo $page + 1; ?>&search=<?php echo urlencode($search); ?>&event_type=<?php echo urlencode($eventType); ?>">Next</a>
            <?php endif; ?>
        </div>
    <?php endif; ?>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
