<?php
$pageTitle = 'Edit Event';
require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/includes/db.php';
requirePermission('events.edit');

$pdo = getDB();
$error = '';
$success = false;

// Get event ID from URL
$eventId = isset($_GET['id']) ? trim($_GET['id']) : '';

if (empty($eventId)) {
    header('HTTP/1.1 404 Not Found');
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Event Not Found | Admin Panel</title>
        <link rel="stylesheet" href="<?php echo ADMIN_BASE_URL; ?>/css/admin.css">
    </head>
    <body>
        <div style="display: flex; align-items: center; justify-content: center; min-height: 100vh; padding: 2rem;">
            <div class="card" style="text-align: center; max-width: 500px;">
                <h1 style="color: var(--danger); margin-bottom: 1rem;">404 - Event Not Found</h1>
                <p style="color: var(--text-medium); margin-bottom: 2rem;">Event ID is required.</p>
                <a href="<?php echo ADMIN_BASE_URL; ?>/events.php" class="btn btn-primary">Back to Events</a>
            </div>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// Fetch existing event - works with any ID format (hex string, candidate_ prefix, etc.)
$stmt = $pdo->prepare("SELECT * FROM events WHERE id = :id LIMIT 1");
$stmt->execute([':id' => $eventId]);
$event = $stmt->fetch();

if (!$event) {
    header('HTTP/1.1 404 Not Found');
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Event Not Found | Admin Panel</title>
        <link rel="stylesheet" href="<?php echo ADMIN_BASE_URL; ?>/css/admin.css">
    </head>
    <body>
        <div style="display: flex; align-items: center; justify-content: center; min-height: 100vh; padding: 2rem;">
            <div class="card" style="text-align: center; max-width: 500px;">
                <h1 style="color: var(--danger); margin-bottom: 1rem;">404 - Event Not Found</h1>
                <p style="color: var(--text-medium); margin-bottom: 0.5rem;">The event with ID <strong><?php echo htmlspecialchars($eventId); ?></strong> was not found.</p>
                <p style="color: var(--text-medium); margin-bottom: 2rem;">It may have been deleted or the ID is incorrect.</p>
                <a href="<?php echo ADMIN_BASE_URL; ?>/events.php" class="btn btn-primary">Back to Events</a>
            </div>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Get and sanitize form data
        $name = trim($_POST['name'] ?? '');
        $description = trim($_POST['description'] ?? '');
        $eventType = trim($_POST['event_type'] ?? 'other');
        $startDate = !empty($_POST['start_date']) ? $_POST['start_date'] : null;
        $endDate = !empty($_POST['end_date']) ? $_POST['end_date'] : null;
        $venue = trim($_POST['venue'] ?? '');
        $city = trim($_POST['city'] ?? '');
        $country = trim($_POST['country'] ?? '');
        $attendeesRange = trim($_POST['attendees_range'] ?? '');
        $ticketAvailable = trim($_POST['ticket_available'] ?? 'no');
        $pricing = trim($_POST['pricing'] ?? '');
        $attendeeProfiles = trim($_POST['attendee_profiles'] ?? '');
        $sponsors = trim($_POST['sponsors'] ?? '');
        $websiteUrl = trim($_POST['website_url'] ?? '');
        
        // Validation
        if (empty($name)) {
            throw new Exception('Event name is required');
        }
        
        // Validate dates
        if ($startDate && !strtotime($startDate)) {
            throw new Exception('Invalid start date format');
        }
        if ($endDate && !strtotime($endDate)) {
            throw new Exception('Invalid end date format');
        }
        if ($startDate && $endDate && strtotime($endDate) < strtotime($startDate)) {
            throw new Exception('End date must be after start date');
        }
        
        // Validate URL if provided
        if ($websiteUrl && !filter_var($websiteUrl, FILTER_VALIDATE_URL)) {
            throw new Exception('Invalid website URL format');
        }
        
        // Prepare SQL update
        $sql = "UPDATE events SET
            name = :name,
            description = :description,
            event_type = :event_type,
            start_date = :start_date,
            end_date = :end_date,
            venue = :venue,
            city = :city,
            country = :country,
            attendees_range = :attendees_range,
            ticket_available = :ticket_available,
            pricing = :pricing,
            attendee_profiles = :attendee_profiles,
            sponsors = :sponsors,
            website_url = :website_url,
            updated_at = NOW()
        WHERE id = :id";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':id' => $eventId,
            ':name' => $name,
            ':description' => $description ?: null,
            ':event_type' => $eventType,
            ':start_date' => $startDate ?: null,
            ':end_date' => $endDate ?: null,
            ':venue' => $venue ?: null,
            ':city' => $city ?: null,
            ':country' => $country ?: null,
            ':attendees_range' => $attendeesRange ?: null,
            ':ticket_available' => $ticketAvailable,
            ':pricing' => $pricing ?: null,
            ':attendee_profiles' => $attendeeProfiles ?: null,
            ':sponsors' => $sponsors ?: null,
            ':website_url' => $websiteUrl ?: null,
        ]);
        
        // Success - redirect to events list
        $_SESSION['success_message'] = 'Event updated successfully!';
        header('Location: ' . ADMIN_BASE_URL . '/events.php');
        exit;
        
    } catch (Exception $e) {
        $error = $e->getMessage();
        error_log('Error updating event: ' . $e->getMessage());
    }
}

// Get event types for dropdown
$eventTypes = [
    'conference' => 'Conference',
    'exhibition' => 'Exhibition',
    'summit' => 'Summit',
    'festival' => 'Festival',
    'trade-show' => 'Trade Show',
    'concert' => 'Concert',
    'sports' => 'Sports',
    'workshop' => 'Workshop',
    'networking' => 'Networking',
    'other' => 'Other'
];

// Use POST data if available (after validation error), otherwise use existing event data
$formData = [
    'name' => $_POST['name'] ?? $event['name'] ?? '',
    'description' => $_POST['description'] ?? $event['description'] ?? '',
    'event_type' => $_POST['event_type'] ?? $event['event_type'] ?? 'other',
    'start_date' => $_POST['start_date'] ?? ($event['start_date'] ? date('Y-m-d', strtotime($event['start_date'])) : ''),
    'end_date' => $_POST['end_date'] ?? ($event['end_date'] ? date('Y-m-d', strtotime($event['end_date'])) : ''),
    'venue' => $_POST['venue'] ?? $event['venue'] ?? '',
    'city' => $_POST['city'] ?? $event['city'] ?? '',
    'country' => $_POST['country'] ?? $event['country'] ?? '',
    'attendees_range' => $_POST['attendees_range'] ?? $event['attendees_range'] ?? '',
    'ticket_available' => $_POST['ticket_available'] ?? $event['ticket_available'] ?? 'no',
    'pricing' => $_POST['pricing'] ?? $event['pricing'] ?? '',
    'attendee_profiles' => $_POST['attendee_profiles'] ?? $event['attendee_profiles'] ?? '',
    'sponsors' => $_POST['sponsors'] ?? $event['sponsors'] ?? '',
    'website_url' => $_POST['website_url'] ?? $event['website_url'] ?? '',
];
?>

<div class="page-header">
    <h1 class="page-title">Edit Event</h1>
    <p class="page-subtitle">Update event information</p>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger" style="background: rgba(231, 76, 60, 0.1); color: var(--danger); padding: 1rem 1.25rem; border-radius: 10px; margin-bottom: 1.5rem; border-left: 4px solid var(--danger);">
        <?php echo htmlspecialchars($error); ?>
    </div>
<?php endif; ?>

<?php if (isset($_SESSION['success_message'])): ?>
    <div class="alert alert-success" style="background: rgba(39, 174, 96, 0.1); color: var(--success); padding: 1rem 1.25rem; border-radius: 10px; margin-bottom: 1.5rem; border-left: 4px solid var(--success);">
        <?php echo htmlspecialchars($_SESSION['success_message']); unset($_SESSION['success_message']); ?>
    </div>
<?php endif; ?>

<div class="card">
    <form method="POST" action="" id="eventForm">
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1.5rem; margin-bottom: 1.5rem;">
            <!-- Event Name -->
            <div class="form-group" style="grid-column: 1 / -1;">
                <label class="form-label">Event Name <span style="color: var(--danger);">*</span></label>
                <input type="text" name="name" class="form-control" required 
                       value="<?php echo htmlspecialchars($formData['name']); ?>" 
                       placeholder="e.g., GITEX Global 2024">
            </div>
            
            <!-- Event Type -->
            <div class="form-group">
                <label class="form-label">Event Type</label>
                <select name="event_type" class="form-control" required>
                    <?php foreach ($eventTypes as $value => $label): ?>
                        <option value="<?php echo htmlspecialchars($value); ?>" 
                                <?php echo ($formData['event_type'] === $value) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($label); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <!-- Start Date -->
            <div class="form-group">
                <label class="form-label">Start Date</label>
                <input type="date" name="start_date" class="form-control" 
                       value="<?php echo htmlspecialchars($formData['start_date']); ?>">
            </div>
            
            <!-- End Date -->
            <div class="form-group">
                <label class="form-label">End Date</label>
                <input type="date" name="end_date" class="form-control" 
                       value="<?php echo htmlspecialchars($formData['end_date']); ?>">
            </div>
            
            <!-- Venue -->
            <div class="form-group">
                <label class="form-label">Venue</label>
                <input type="text" name="venue" class="form-control" 
                       value="<?php echo htmlspecialchars($formData['venue']); ?>" 
                       placeholder="e.g., Dubai World Trade Centre">
            </div>
            
            <!-- City -->
            <div class="form-group">
                <label class="form-label">City</label>
                <input type="text" name="city" class="form-control" 
                       value="<?php echo htmlspecialchars($formData['city']); ?>" 
                       placeholder="e.g., Dubai">
            </div>
            
            <!-- Country -->
            <div class="form-group">
                <label class="form-label">Country</label>
                <input type="text" name="country" class="form-control" 
                       value="<?php echo htmlspecialchars($formData['country']); ?>" 
                       placeholder="e.g., UAE">
            </div>
            
            <!-- Attendees Range -->
            <div class="form-group">
                <label class="form-label">Attendees Range</label>
                <input type="text" name="attendees_range" class="form-control" 
                       value="<?php echo htmlspecialchars($formData['attendees_range']); ?>" 
                       placeholder="e.g., 5000-10000 or 180,000+">
            </div>
            
            <!-- Ticket Available -->
            <div class="form-group">
                <label class="form-label">Tickets Available</label>
                <select name="ticket_available" class="form-control">
                    <option value="yes" <?php echo ($formData['ticket_available'] === 'yes') ? 'selected' : ''; ?>>Yes</option>
                    <option value="no" <?php echo ($formData['ticket_available'] === 'no') ? 'selected' : ''; ?>>No</option>
                </select>
            </div>
            
            <!-- Pricing -->
            <div class="form-group">
                <label class="form-label">Pricing</label>
                <input type="text" name="pricing" class="form-control" 
                       value="<?php echo htmlspecialchars($formData['pricing']); ?>" 
                       placeholder="e.g., From AED 350">
            </div>
            
            <!-- Website URL -->
            <div class="form-group" style="grid-column: 1 / -1;">
                <label class="form-label">Website URL</label>
                <input type="url" name="website_url" class="form-control" 
                       value="<?php echo htmlspecialchars($formData['website_url']); ?>" 
                       placeholder="https://example.com">
            </div>
            
            <!-- Description -->
            <div class="form-group" style="grid-column: 1 / -1;">
                <label class="form-label">Description</label>
                <textarea name="description" class="form-control" rows="4" 
                          placeholder="Detailed description of the event..."><?php echo htmlspecialchars($formData['description']); ?></textarea>
            </div>
            
            <!-- Attendee Profiles -->
            <div class="form-group" style="grid-column: 1 / -1;">
                <label class="form-label">Attendee Profiles</label>
                <input type="text" name="attendee_profiles" class="form-control" 
                       value="<?php echo htmlspecialchars($formData['attendee_profiles']); ?>" 
                       placeholder="e.g., Tech executives, startups, investors">
            </div>
            
            <!-- Sponsors -->
            <div class="form-group" style="grid-column: 1 / -1;">
                <label class="form-label">Sponsors</label>
                <input type="text" name="sponsors" class="form-control" 
                       value="<?php echo htmlspecialchars($formData['sponsors']); ?>" 
                       placeholder="Comma-separated list of sponsors">
            </div>
        </div>
        
        <div style="display: flex; gap: 1rem; justify-content: flex-end; padding-top: 1.5rem; border-top: 1px solid var(--border-color);">
            <a href="<?php echo ADMIN_BASE_URL; ?>/events.php" class="btn btn-secondary">Cancel</a>
            <button type="submit" class="btn btn-primary">Update Event</button>
        </div>
    </form>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
