<?php
$pageTitle = 'Add New Event';
require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/includes/db.php';
requirePermission('events.create');

$pdo = getDB();
$error = '';
$success = false;

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Generate unique ID for the event
        $eventId = bin2hex(random_bytes(16)); // 32 character hex string
        
        // Get and sanitize form data
        $name = trim($_POST['name'] ?? '');
        $description = trim($_POST['description'] ?? '');
        $eventType = trim($_POST['event_type'] ?? 'other');
        $startDate = !empty($_POST['start_date']) ? $_POST['start_date'] : null;
        $endDate = !empty($_POST['end_date']) ? $_POST['end_date'] : null;
        $venue = trim($_POST['venue'] ?? '');
        $city = trim($_POST['city'] ?? '');
        $country = trim($_POST['country'] ?? '');
        $attendeesRange = trim($_POST['attendees_range'] ?? '');
        $ticketAvailable = trim($_POST['ticket_available'] ?? 'no');
        $pricing = trim($_POST['pricing'] ?? '');
        $attendeeProfiles = trim($_POST['attendee_profiles'] ?? '');
        $sponsors = trim($_POST['sponsors'] ?? '');
        $websiteUrl = trim($_POST['website_url'] ?? '');
        
        // Validation
        if (empty($name)) {
            throw new Exception('Event name is required');
        }
        
        // Validate dates
        if ($startDate && !strtotime($startDate)) {
            throw new Exception('Invalid start date format');
        }
        if ($endDate && !strtotime($endDate)) {
            throw new Exception('Invalid end date format');
        }
        if ($startDate && $endDate && strtotime($endDate) < strtotime($startDate)) {
            throw new Exception('End date must be after start date');
        }
        
        // Validate URL if provided
        if ($websiteUrl && !filter_var($websiteUrl, FILTER_VALIDATE_URL)) {
            throw new Exception('Invalid website URL format');
        }
        
        // Prepare SQL insert
        $sql = "INSERT INTO events (
            id, name, description, event_type, start_date, end_date,
            venue, city, country, attendees_range, ticket_available,
            pricing, attendee_profiles, sponsors, website_url,
            created_at, updated_at
        ) VALUES (
            :id, :name, :description, :event_type, :start_date, :end_date,
            :venue, :city, :country, :attendees_range, :ticket_available,
            :pricing, :attendee_profiles, :sponsors, :website_url,
            NOW(), NOW()
        )";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':id' => $eventId,
            ':name' => $name,
            ':description' => $description ?: null,
            ':event_type' => $eventType,
            ':start_date' => $startDate ?: null,
            ':end_date' => $endDate ?: null,
            ':venue' => $venue ?: null,
            ':city' => $city ?: null,
            ':country' => $country ?: null,
            ':attendees_range' => $attendeesRange ?: null,
            ':ticket_available' => $ticketAvailable,
            ':pricing' => $pricing ?: null,
            ':attendee_profiles' => $attendeeProfiles ?: null,
            ':sponsors' => $sponsors ?: null,
            ':website_url' => $websiteUrl ?: null,
        ]);
        
        // Success - redirect to events list
        $_SESSION['success_message'] = 'Event created successfully!';
        header('Location: ' . ADMIN_BASE_URL . '/events.php');
        exit;
        
    } catch (Exception $e) {
        $error = $e->getMessage();
        error_log('Error adding event: ' . $e->getMessage());
    }
}

// Get event types for dropdown
$eventTypes = [
    'conference' => 'Conference',
    'exhibition' => 'Exhibition',
    'summit' => 'Summit',
    'festival' => 'Festival',
    'trade-show' => 'Trade Show',
    'concert' => 'Concert',
    'sports' => 'Sports',
    'workshop' => 'Workshop',
    'networking' => 'Networking',
    'other' => 'Other'
];
?>

<div class="page-header">
    <h1 class="page-title">Add New Event</h1>
    <p class="page-subtitle">Create a new event in the system</p>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger" style="background: rgba(231, 76, 60, 0.1); color: var(--danger); padding: 1rem 1.25rem; border-radius: 10px; margin-bottom: 1.5rem; border-left: 4px solid var(--danger);">
        <?php echo htmlspecialchars($error); ?>
    </div>
<?php endif; ?>

<?php if (isset($_SESSION['success_message'])): ?>
    <div class="alert alert-success" style="background: rgba(39, 174, 96, 0.1); color: var(--success); padding: 1rem 1.25rem; border-radius: 10px; margin-bottom: 1.5rem; border-left: 4px solid var(--success);">
        <?php echo htmlspecialchars($_SESSION['success_message']); unset($_SESSION['success_message']); ?>
    </div>
<?php endif; ?>

<div class="card">
    <form method="POST" action="" id="eventForm">
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1.5rem; margin-bottom: 1.5rem;">
            <!-- Event Name -->
            <div class="form-group" style="grid-column: 1 / -1;">
                <label class="form-label">Event Name <span style="color: var(--danger);">*</span></label>
                <input type="text" name="name" class="form-control" required 
                       value="<?php echo htmlspecialchars($_POST['name'] ?? ''); ?>" 
                       placeholder="e.g., GITEX Global 2024">
            </div>
            
            <!-- Event Type -->
            <div class="form-group">
                <label class="form-label">Event Type</label>
                <select name="event_type" class="form-control" required>
                    <?php foreach ($eventTypes as $value => $label): ?>
                        <option value="<?php echo htmlspecialchars($value); ?>" 
                                <?php echo (($_POST['event_type'] ?? 'other') === $value) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($label); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <!-- Start Date -->
            <div class="form-group">
                <label class="form-label">Start Date</label>
                <input type="date" name="start_date" class="form-control" 
                       value="<?php echo htmlspecialchars($_POST['start_date'] ?? ''); ?>">
            </div>
            
            <!-- End Date -->
            <div class="form-group">
                <label class="form-label">End Date</label>
                <input type="date" name="end_date" class="form-control" 
                       value="<?php echo htmlspecialchars($_POST['end_date'] ?? ''); ?>">
            </div>
            
            <!-- Venue -->
            <div class="form-group">
                <label class="form-label">Venue</label>
                <input type="text" name="venue" class="form-control" 
                       value="<?php echo htmlspecialchars($_POST['venue'] ?? ''); ?>" 
                       placeholder="e.g., Dubai World Trade Centre">
            </div>
            
            <!-- City -->
            <div class="form-group">
                <label class="form-label">City</label>
                <input type="text" name="city" class="form-control" 
                       value="<?php echo htmlspecialchars($_POST['city'] ?? ''); ?>" 
                       placeholder="e.g., Dubai">
            </div>
            
            <!-- Country -->
            <div class="form-group">
                <label class="form-label">Country</label>
                <input type="text" name="country" class="form-control" 
                       value="<?php echo htmlspecialchars($_POST['country'] ?? ''); ?>" 
                       placeholder="e.g., UAE">
            </div>
            
            <!-- Attendees Range -->
            <div class="form-group">
                <label class="form-label">Attendees Range</label>
                <input type="text" name="attendees_range" class="form-control" 
                       value="<?php echo htmlspecialchars($_POST['attendees_range'] ?? ''); ?>" 
                       placeholder="e.g., 5000-10000 or 180,000+">
            </div>
            
            <!-- Ticket Available -->
            <div class="form-group">
                <label class="form-label">Tickets Available</label>
                <select name="ticket_available" class="form-control">
                    <option value="yes" <?php echo (($_POST['ticket_available'] ?? 'no') === 'yes') ? 'selected' : ''; ?>>Yes</option>
                    <option value="no" <?php echo (($_POST['ticket_available'] ?? 'no') === 'no') ? 'selected' : ''; ?>>No</option>
                </select>
            </div>
            
            <!-- Pricing -->
            <div class="form-group">
                <label class="form-label">Pricing</label>
                <input type="text" name="pricing" class="form-control" 
                       value="<?php echo htmlspecialchars($_POST['pricing'] ?? ''); ?>" 
                       placeholder="e.g., From AED 350">
            </div>
            
            <!-- Website URL -->
            <div class="form-group" style="grid-column: 1 / -1;">
                <label class="form-label">Website URL</label>
                <input type="url" name="website_url" class="form-control" 
                       value="<?php echo htmlspecialchars($_POST['website_url'] ?? ''); ?>" 
                       placeholder="https://example.com">
            </div>
            
            <!-- Description -->
            <div class="form-group" style="grid-column: 1 / -1;">
                <label class="form-label">Description</label>
                <textarea name="description" class="form-control" rows="4" 
                          placeholder="Detailed description of the event..."><?php echo htmlspecialchars($_POST['description'] ?? ''); ?></textarea>
            </div>
            
            <!-- Attendee Profiles -->
            <div class="form-group" style="grid-column: 1 / -1;">
                <label class="form-label">Attendee Profiles</label>
                <input type="text" name="attendee_profiles" class="form-control" 
                       value="<?php echo htmlspecialchars($_POST['attendee_profiles'] ?? ''); ?>" 
                       placeholder="e.g., Tech executives, startups, investors">
            </div>
            
            <!-- Sponsors -->
            <div class="form-group" style="grid-column: 1 / -1;">
                <label class="form-label">Sponsors</label>
                <input type="text" name="sponsors" class="form-control" 
                       value="<?php echo htmlspecialchars($_POST['sponsors'] ?? ''); ?>" 
                       placeholder="Comma-separated list of sponsors">
            </div>
        </div>
        
        <div style="display: flex; gap: 1rem; justify-content: flex-end; padding-top: 1.5rem; border-top: 1px solid var(--border-color);">
            <a href="<?php echo ADMIN_BASE_URL; ?>/events.php" class="btn btn-secondary">Cancel</a>
            <button type="submit" class="btn btn-primary">Create Event</button>
        </div>
    </form>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
