<?php
$pageTitle = 'Customers Management';
require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/includes/db.php';
requirePermission('customers.view');

$pdo = getDB();

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$perPage = ITEMS_PER_PAGE;
$offset = ($page - 1) * $perPage;

// Search and filters
$search = $_GET['search'] ?? '';
$provider = $_GET['provider'] ?? '';

$where = ['1=1'];
$params = [];

if ($search) {
    $where[] = "(full_name LIKE :search OR email LIKE :search OR first_name LIKE :search OR last_name LIKE :search)";
    $params[':search'] = '%' . $search . '%';
}

if ($provider) {
    $where[] = "provider = :provider";
    $params[':provider'] = $provider;
}

$whereClause = implode(' AND ', $where);

// Get total count
$countStmt = $pdo->prepare("SELECT COUNT(*) as total FROM customers WHERE $whereClause");
$countStmt->execute($params);
$totalItems = $countStmt->fetch()['total'];
$totalPages = ceil($totalItems / $perPage);

// Get customers
$sql = "SELECT * FROM customers WHERE $whereClause ORDER BY created_at DESC LIMIT :limit OFFSET :offset";
$stmt = $pdo->prepare($sql);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$customers = $stmt->fetchAll();
?>

<div class="page-header">
    <h1 class="page-title">Customers Management</h1>
    <p class="page-subtitle">Manage all registered customers</p>
</div>

<!-- Search and Filters -->
<div class="card">
    <form method="GET" action="" style="display: flex; gap: 1rem; flex-wrap: wrap; align-items: end;">
        <div class="form-group" style="flex: 1; min-width: 200px;">
            <label class="form-label">Search</label>
            <input type="text" name="search" class="form-control" placeholder="Search by name or email..." value="<?php echo htmlspecialchars($search); ?>">
        </div>
        
        <div class="form-group" style="min-width: 150px;">
            <label class="form-label">Provider</label>
            <select name="provider" class="form-control">
                <option value="">All Providers</option>
                <option value="google" <?php echo $provider === 'google' ? 'selected' : ''; ?>>Google</option>
                <option value="facebook" <?php echo $provider === 'facebook' ? 'selected' : ''; ?>>Facebook</option>
                <option value="linkedin" <?php echo $provider === 'linkedin' ? 'selected' : ''; ?>>LinkedIn</option>
            </select>
        </div>
        
        <div class="form-group">
            <button type="submit" class="btn btn-primary">Filter</button>
            <a href="customers.php" class="btn btn-secondary">Reset</a>
        </div>
    </form>
</div>

<!-- Customers Table -->
<div class="card">
    <div class="card-header">
        <h2 class="card-title">Customers (<?php echo number_format($totalItems); ?>)</h2>
    </div>
    
    <div class="table-container">
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Name</th>
                    <th>Email</th>
                    <th>Provider</th>
                    <th>Location</th>
                    <th>Joined</th>
                    <th>Last Login</th>
                    <th>Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($customers)): ?>
                    <tr>
                        <td colspan="9" style="text-align: center; color: var(--text-medium); padding: 2rem;">No customers found</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($customers as $customer): ?>
                        <tr>
                            <td><?php echo $customer['id']; ?></td>
                            <td>
                                <?php if ($customer['profile_picture']): ?>
                                    <img src="<?php echo htmlspecialchars($customer['profile_picture']); ?>" 
                                         alt="Avatar" style="width: 32px; height: 32px; border-radius: 50%; margin-right: 0.5rem; vertical-align: middle;">
                                <?php endif; ?>
                                <strong><?php echo htmlspecialchars($customer['full_name'] ?? ($customer['first_name'] . ' ' . $customer['last_name']) ?? 'N/A'); ?></strong>
                            </td>
                            <td><?php echo htmlspecialchars($customer['email']); ?></td>
                            <td><span class="badge badge-info"><?php echo ucfirst($customer['provider']); ?></span></td>
                            <td>
                                <?php 
                                $location = [];
                                if ($customer['city']) $location[] = $customer['city'];
                                if ($customer['country']) $location[] = $customer['country'];
                                echo htmlspecialchars(implode(', ', $location) ?: 'N/A');
                                ?>
                            </td>
                            <td><?php echo date('M d, Y', strtotime($customer['created_at'])); ?></td>
                            <td><?php echo $customer['last_login'] ? date('M d, Y', strtotime($customer['last_login'])) : 'Never'; ?></td>
                            <td>
                                <?php if ($customer['is_active']): ?>
                                    <span class="badge badge-success">Active</span>
                                <?php else: ?>
                                    <span class="badge badge-danger">Inactive</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <div style="display: flex; gap: 0.5rem;">
                                    <a href="customer-view.php?id=<?php echo $customer['id']; ?>" class="btn btn-secondary btn-sm">View</a>
                                    <?php if (hasPermission('customers.edit')): ?>
                                        <a href="customer-edit.php?id=<?php echo $customer['id']; ?>" class="btn btn-secondary btn-sm">Edit</a>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Pagination -->
    <?php if ($totalPages > 1): ?>
        <div class="pagination">
            <?php if ($page > 1): ?>
                <a href="?page=<?php echo $page - 1; ?>&search=<?php echo urlencode($search); ?>&provider=<?php echo urlencode($provider); ?>">Previous</a>
            <?php endif; ?>
            
            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                <a href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&provider=<?php echo urlencode($provider); ?>" 
                   class="<?php echo $i === $page ? 'active' : ''; ?>">
                    <?php echo $i; ?>
                </a>
            <?php endfor; ?>
            
            <?php if ($page < $totalPages): ?>
                <a href="?page=<?php echo $page + 1; ?>&search=<?php echo urlencode($search); ?>&provider=<?php echo urlencode($provider); ?>">Next</a>
            <?php endif; ?>
        </div>
    <?php endif; ?>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
